<?php

namespace Tests\Unit;

use PHPUnit\Framework\TestCase;

class OrderApprovalFlowCalculationTest extends TestCase
{
    /**
     * Test approved price calculation with string inputs
     * This test validates that the TypeError is fixed for approved_price calculation
     */
    public function test_approved_price_calculation_with_string_inputs(): void
    {
        // Simulate values that would come from form fields
        $offerPrice = '1000.50'; // string from form input
        $additionalFees = '250.75'; // string from form input

        // Apply the same type casting as in the fixed code
        $offerPrice = (float) ($offerPrice ?? 0);
        $additionalFees = (float) ($additionalFees ?? 0);

        $approvedPrice = $offerPrice + $additionalFees;

        // Expected result: 1000.50 + 250.75 = 1251.25
        $this->assertEquals(1251.25, $approvedPrice);
    }

    /**
     * Test operation cost calculation with string inputs
     * This test validates that the TypeError is fixed for operation_cost calculation
     */
    public function test_operation_cost_calculation_with_string_inputs(): void
    {
        // Simulate values that would come from form fields
        $offerPrice = '2000.00'; // string from form input
        $additionalFees = '500.00'; // string from form input
        $operationCostPercentage = '15.5'; // string from form input

        // Apply the same type casting as in the fixed code
        $offerPrice = (float) ($offerPrice ?? 0);
        $additionalFees = (float) ($additionalFees ?? 0);
        $operationCostPercentage = (float) ($operationCostPercentage ?? 0);

        $operationCost = ($offerPrice + $additionalFees) * $operationCostPercentage / 100;

        // Expected result: (2000.00 + 500.00) * 15.5 / 100 = 387.5
        $this->assertEquals(387.5, $operationCost);
    }

    /**
     * Test edge cases for both calculations
     */
    public function test_calculation_edge_cases(): void
    {
        // Test with null values
        $offerPrice = (float) (null ?? 0);
        $additionalFees = (float) (null ?? 0);
        $operationCostPercentage = (float) (null ?? 0);

        $approvedPrice = $offerPrice + $additionalFees;
        $this->assertEquals(0.0, $approvedPrice);

        // Operation cost should return early when percentage is 0
        $this->assertEquals(0.0, $operationCostPercentage);

        // Test with empty strings
        $offerPrice = (float) ('' ?? 0);
        $additionalFees = (float) ('' ?? 0);
        $operationCostPercentage = (float) ('' ?? 0);

        $approvedPrice = $offerPrice + $additionalFees;
        $this->assertEquals(0.0, $approvedPrice);

        // Test with mixed numeric and string values
        $offerPrice = (float) (500.25 ?? 0);
        $additionalFees = (float) ('100.75' ?? 0);
        $operationCostPercentage = (float) ('10' ?? 0);

        $approvedPrice = $offerPrice + $additionalFees;
        $operationCost = $approvedPrice * $operationCostPercentage / 100;

        $this->assertEquals(601.0, $approvedPrice);
        $this->assertEquals(60.1, $operationCost);
    }

    /**
     * Test operation cost percentage zero condition
     */
    public function test_operation_cost_zero_percentage(): void
    {
        $operationCostPercentage = (float) (0 ?? 0);

        // The condition checks for === 0, but after casting to float it becomes 0.0
        // which is equal to 0 but not identical (=== is false), so we need == or === 0.0
        $this->assertEquals(0.0, $operationCostPercentage);
        $this->assertTrue($operationCostPercentage == 0);

        // Test with string '0'
        $operationCostPercentage = (float) ('0' ?? 0);
        $this->assertEquals(0.0, $operationCostPercentage);
        $this->assertTrue($operationCostPercentage == 0);
    }
}
