<?php

namespace Tests\Unit;

use App\Models\DonorFinancialPortfolio;
use PHPUnit\Framework\TestCase;

class FinancialPortfolioFeatureTest extends TestCase
{
    /**
     * Test that total operating value is calculated correctly
     * Based on issue requirements: نسبة التشغيل (7%) should calculate إجمالي قيمة التشغيل (105,000)
     */
    public function test_total_operating_value_calculation(): void
    {
        $portfolio = new DonorFinancialPortfolio([
            'name' => 'Test Portfolio',
            'balance' => 1500000, // 1.5 million
            'operating_percentage' => 7.0, // 7%
        ]);

        // Expected: 7% of 1,500,000 = 105,000
        $expected = 105000;
        $actual = $portfolio->total_operating_value;

        $this->assertEquals($expected, $actual, 'Total operating value should be calculated as operating_percentage * balance / 100');
    }

    /**
     * Test that all required columns are available as attributes
     */
    public function test_required_columns_are_available(): void
    {
        $portfolio = new DonorFinancialPortfolio([
            'name' => 'Test Portfolio',
            'balance' => 1000000,
            'operating_percentage' => 5.0,
        ]);

        // Required columns from issue:
        // 1. اسم المحفظة
        $this->assertNotNull($portfolio->name);

        // 2. إجمالي قيمة المحفظة
        $this->assertNotNull($portfolio->balance);

        // 4. نسبة التشغيل
        $this->assertNotNull($portfolio->operating_percentage);

        // 5. إجمالي قيمة التشغيل
        $this->assertIsNumeric($portfolio->total_operating_value);

        // Database-dependent attributes would be tested with database (not unit test)
        // 3. المخصص من المحفظة (total_allocated)
        // 6. الرصيد المتبقي (remaining_balance)
        // 7. عدد الطلبات (orders count)
    }

    /**
     * Test edge cases for operating value calculation
     */
    public function test_operating_value_edge_cases(): void
    {
        // Test with null operating percentage
        $portfolio1 = new DonorFinancialPortfolio([
            'name' => 'Test Portfolio 1',
            'balance' => 1000000,
            'operating_percentage' => null,
        ]);
        $this->assertEquals(0, $portfolio1->total_operating_value, 'Should return 0 when operating_percentage is null');

        // Test with null balance
        $portfolio2 = new DonorFinancialPortfolio([
            'name' => 'Test Portfolio 2',
            'balance' => null,
            'operating_percentage' => 5.0,
        ]);
        $this->assertEquals(0, $portfolio2->total_operating_value, 'Should return 0 when balance is null');

        // Test with zero values
        $portfolio3 = new DonorFinancialPortfolio([
            'name' => 'Test Portfolio 3',
            'balance' => 0,
            'operating_percentage' => 5.0,
        ]);
        $this->assertEquals(0, $portfolio3->total_operating_value, 'Should return 0 when balance is 0');
    }

    /**
     * Test that total operating value can be summed across multiple portfolios
     * This validates the functionality added for the dashboard widget
     */
    public function test_total_operating_value_sum_across_portfolios(): void
    {
        // Create test portfolios with different operating values
        $portfolios = collect([
            new DonorFinancialPortfolio([
                'name' => 'Portfolio 1',
                'balance' => 1000000, // 1M
                'operating_percentage' => 5.0, // 5% = 50,000
            ]),
            new DonorFinancialPortfolio([
                'name' => 'Portfolio 2',
                'balance' => 2000000, // 2M
                'operating_percentage' => 7.5, // 7.5% = 150,000
            ]),
            new DonorFinancialPortfolio([
                'name' => 'Portfolio 3',
                'balance' => 500000, // 500K
                'operating_percentage' => 10.0, // 10% = 50,000
            ]),
        ]);

        // Calculate total operating value
        $totalOperatingValue = $portfolios->sum('total_operating_value');

        // Expected: 50,000 + 150,000 + 50,000 = 250,000
        $expected = 250000;

        $this->assertEquals($expected, $totalOperatingValue, 'Total operating value should be the sum of all individual portfolio operating values');
    }
}
