<?php

namespace Tests\Unit;

use App\Models\DonorFinancialPortfolio;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class DonorFinancialPortfolioOperatingPercentageTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Test that operating percentage field is properly included in the model
     */
    public function test_operating_percentage_is_fillable(): void
    {
        $portfolio = DonorFinancialPortfolio::create([
            'name' => 'Test Portfolio',
            'balance' => 100000.00,
            'operating_percentage' => 5.25,
        ]);

        $this->assertEquals('Test Portfolio', $portfolio->name);
        $this->assertEquals(100000.00, $portfolio->balance);
        $this->assertEquals(5.25, $portfolio->operating_percentage);
    }

    /**
     * Test that operating percentage field is properly cast to decimal
     */
    public function test_operating_percentage_casting(): void
    {
        $portfolio = DonorFinancialPortfolio::create([
            'name' => 'Test Portfolio',
            'balance' => 50000.00,
            'operating_percentage' => '7.5',
        ]);

        // Verify the value is cast to a decimal
        $this->assertIsFloat($portfolio->operating_percentage);
        $this->assertEquals(7.5, $portfolio->operating_percentage);
    }

    /**
     * Test that the factory includes operating percentage
     */
    public function test_factory_includes_operating_percentage(): void
    {
        $portfolio = DonorFinancialPortfolio::factory()->create();

        $this->assertNotNull($portfolio->operating_percentage);
        $this->assertIsFloat($portfolio->operating_percentage);
        $this->assertGreaterThanOrEqual(1.5, $portfolio->operating_percentage);
        $this->assertLessThanOrEqual(15.0, $portfolio->operating_percentage);
    }

    /**
     * Test factory states include operating percentage
     */
    public function test_factory_states_include_operating_percentage(): void
    {
        $portfolios = [
            DonorFinancialPortfolio::factory()->largeFund()->create(),
            DonorFinancialPortfolio::factory()->emergencyFund()->create(),
            DonorFinancialPortfolio::factory()->communityFund()->create(),
            DonorFinancialPortfolio::factory()->infrastructureFund()->create(),
        ];

        foreach ($portfolios as $portfolio) {
            $this->assertNotNull($portfolio->operating_percentage);
            $this->assertIsFloat($portfolio->operating_percentage);
            $this->assertGreaterThan(0, $portfolio->operating_percentage);
            $this->assertLessThan(100, $portfolio->operating_percentage);
        }
    }

    /**
     * Test that operating percentage can be null (migration allows nullable)
     */
    public function test_operating_percentage_can_be_null(): void
    {
        $portfolio = DonorFinancialPortfolio::create([
            'name' => 'Test Portfolio',
            'balance' => 75000.00,
            'operating_percentage' => null,
        ]);

        $this->assertNull($portfolio->operating_percentage);
    }
}
