<?php

namespace Tests\Feature;

use App\Models\ProjectCharter;
use App\Models\ProjectGovernance;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ProjectGovernanceTest extends TestCase
{
    use RefreshDatabase;

    /** @test */
    public function it_can_create_a_project_governance()
    {
        // Create a user first (required for project charter)
        $user = User::factory()->create();
        $this->actingAs($user);

        // Create a project charter
        $charter = ProjectCharter::factory()->create();

        // Create a project governance
        $governance = ProjectGovernance::create([
            'charter_id' => $charter->id,
            'name' => 'Project Governance Item',
            'description' => 'This is a description for project governance',
        ]);

        $this->assertDatabaseHas('project_governances', [
            'charter_id' => $charter->id,
            'name' => 'Project Governance Item',
            'description' => 'This is a description for project governance',
            'created_by' => $user->id,
        ]);

        // Test relationships
        $this->assertEquals($charter->id, $governance->charter->id);
        $this->assertTrue($charter->projectGovernances->contains($governance));
        $this->assertEquals($user->id, $governance->creator->id);
    }

    /** @test */
    public function it_creates_project_governance_with_nullable_fields()
    {
        $user = User::factory()->create();
        $this->actingAs($user);

        $charter = ProjectCharter::factory()->create();

        // Create governance with only required fields
        $governance = ProjectGovernance::create([
            'charter_id' => $charter->id,
            'name' => 'Required Name Only',
        ]);

        $this->assertDatabaseHas('project_governances', [
            'charter_id' => $charter->id,
            'name' => 'Required Name Only',
            'description' => null,
            'created_by' => $user->id,
        ]);
    }

    /** @test */
    public function it_requires_name_field()
    {
        $user = User::factory()->create();
        $this->actingAs($user);

        $charter = ProjectCharter::factory()->create();

        $this->expectException(\Illuminate\Database\QueryException::class);

        ProjectGovernance::create([
            'charter_id' => $charter->id,
            'description' => 'Description without name',
        ]);
    }

    /** @test */
    public function it_can_have_multiple_governance_items_per_charter()
    {
        $user = User::factory()->create();
        $this->actingAs($user);

        $charter = ProjectCharter::factory()->create();

        // Create multiple governance items
        $governance1 = ProjectGovernance::create([
            'charter_id' => $charter->id,
            'name' => 'First Governance Item',
            'description' => 'First description',
        ]);

        $governance2 = ProjectGovernance::create([
            'charter_id' => $charter->id,
            'name' => 'Second Governance Item',
            'description' => 'Second description',
        ]);

        $this->assertEquals(2, $charter->projectGovernances()->count());
        $this->assertTrue($charter->projectGovernances->contains($governance1));
        $this->assertTrue($charter->projectGovernances->contains($governance2));
    }
}
