<?php

namespace Tests\Feature;

use App\Models\ProjectCharter;
use App\Models\ProjectCharterChangeRequest;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ProjectCharterChangeRequestSimpleTest extends TestCase
{
    use RefreshDatabase;

    public function test_change_request_has_correct_status_constants(): void
    {
        $this->assertEquals('pending', ProjectCharterChangeRequest::STATUS_PENDING);
        $this->assertEquals('first_approved', ProjectCharterChangeRequest::STATUS_FIRST_APPROVED);
        $this->assertEquals('approved', ProjectCharterChangeRequest::STATUS_APPROVED);
        $this->assertEquals('rejected', ProjectCharterChangeRequest::STATUS_REJECTED);
    }

    public function test_change_request_can_be_created_with_new_fields(): void
    {
        $user = User::factory()->create();
        $charter = ProjectCharter::factory()->create();

        $changeRequest = ProjectCharterChangeRequest::factory()->create([
            'project_charter_id' => $charter->id,
            'requested_by' => $user->id,
            'status' => ProjectCharterChangeRequest::STATUS_PENDING,
        ]);

        $this->assertInstanceOf(ProjectCharterChangeRequest::class, $changeRequest);
        $this->assertEquals(ProjectCharterChangeRequest::STATUS_PENDING, $changeRequest->status);
        $this->assertEquals($charter->id, $changeRequest->project_charter_id);
        $this->assertEquals($user->id, $changeRequest->requested_by);
    }

    public function test_change_request_can_transition_through_approval_stages(): void
    {
        $changeRequest = ProjectCharterChangeRequest::factory()->pending()->create();
        $firstApprover = User::factory()->create();
        $finalApprover = User::factory()->create();

        // Test first approval
        $changeRequest->status = ProjectCharterChangeRequest::STATUS_FIRST_APPROVED;
        $changeRequest->first_approved_by = $firstApprover->id;
        $changeRequest->first_approval_notes = 'First approval notes';
        $changeRequest->first_approved_at = now();
        $changeRequest->save();

        $this->assertEquals(ProjectCharterChangeRequest::STATUS_FIRST_APPROVED, $changeRequest->status);
        $this->assertEquals($firstApprover->id, $changeRequest->first_approved_by);

        // Test final approval
        $changeRequest->status = ProjectCharterChangeRequest::STATUS_APPROVED;
        $changeRequest->final_approved_by = $finalApprover->id;
        $changeRequest->final_approval_notes = 'Final approval notes';
        $changeRequest->final_approved_at = now();
        $changeRequest->save();

        $this->assertEquals(ProjectCharterChangeRequest::STATUS_APPROVED, $changeRequest->status);
        $this->assertEquals($finalApprover->id, $changeRequest->final_approved_by);
    }

    public function test_change_request_relationships_work(): void
    {
        $firstApprover = User::factory()->create(['name' => 'First Approver']);
        $finalApprover = User::factory()->create(['name' => 'Final Approver']);

        $changeRequest = ProjectCharterChangeRequest::factory()->approved()->create([
            'first_approved_by' => $firstApprover->id,
            'final_approved_by' => $finalApprover->id,
        ]);

        $this->assertEquals('First Approver', $changeRequest->firstApprovedBy->name);
        $this->assertEquals('Final Approver', $changeRequest->finalApprovedBy->name);
    }

    public function test_get_statuses_method_includes_new_status(): void
    {
        $statuses = ProjectCharterChangeRequest::getStatuses();

        $this->assertArrayHasKey(ProjectCharterChangeRequest::STATUS_PENDING, $statuses);
        $this->assertArrayHasKey(ProjectCharterChangeRequest::STATUS_FIRST_APPROVED, $statuses);
        $this->assertArrayHasKey(ProjectCharterChangeRequest::STATUS_APPROVED, $statuses);
        $this->assertArrayHasKey(ProjectCharterChangeRequest::STATUS_REJECTED, $statuses);
    }

    public function test_factory_states_work_correctly(): void
    {
        $pendingRequest = ProjectCharterChangeRequest::factory()->pending()->create();
        $firstApprovedRequest = ProjectCharterChangeRequest::factory()->firstApproved()->create();
        $approvedRequest = ProjectCharterChangeRequest::factory()->approved()->create();
        $rejectedRequest = ProjectCharterChangeRequest::factory()->rejected()->create();

        $this->assertEquals(ProjectCharterChangeRequest::STATUS_PENDING, $pendingRequest->status);
        $this->assertEquals(ProjectCharterChangeRequest::STATUS_FIRST_APPROVED, $firstApprovedRequest->status);
        $this->assertEquals(ProjectCharterChangeRequest::STATUS_APPROVED, $approvedRequest->status);
        $this->assertEquals(ProjectCharterChangeRequest::STATUS_REJECTED, $rejectedRequest->status);

        // Test that approved requests have both approval fields
        $this->assertNotNull($approvedRequest->first_approved_by);
        $this->assertNotNull($approvedRequest->final_approved_by);
        $this->assertNotNull($approvedRequest->first_approved_at);
        $this->assertNotNull($approvedRequest->final_approved_at);
    }
}
