<?php

namespace Tests\Feature;

use App\Models\City;
use App\Models\State;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class OrderLocationCoordinatesTest extends TestCase
{
    use RefreshDatabase;

    private User $user;

    protected function setUp(): void
    {
        parent::setUp();

        // Create an admin user for testing
        $this->user = User::factory()->create();
        $this->actingAs($this->user);
    }

    public function test_order_resource_location_information_fields_structure(): void
    {
        // Test that the location information fields include the expected structure
        $fields = \App\Filament\Resources\OrderResource::getLocationInformationFields();

        // Verify we have the expected number of fields
        $this->assertCount(4, $fields); // state_id, city_id, centers, location + hidden lat/lng

        // Find the city_id field and verify it has afterStateUpdated callback
        $cityField = collect($fields)->first(fn ($field) => $field->getName() === 'city_id');
        $this->assertNotNull($cityField);

        // Test that the field is live and has afterStateUpdated
        $this->assertTrue($cityField->isLive());
        $this->assertNotNull($cityField->getAfterStateUpdated());
    }

    public function test_location_coordinates_service_integration_with_real_data(): void
    {
        // Create test data
        $state = State::factory()->create([
            'name' => ['ar' => 'الرياض', 'en' => 'Riyadh'],
            'lat' => 24.7136,
            'lng' => 46.6753,
        ]);

        $city = City::factory()->create([
            'name' => ['ar' => 'جدة', 'en' => 'Jeddah'],
            'state_id' => $state->id,
            'lat' => 21.2854,
            'lng' => 39.2376,
        ]);

        // Test the service integration
        $service = app(\App\Services\LocationCoordinatesService::class);

        // Test city coordinates
        $cityCoords = $service->getCityCoordinates($city->id);
        $this->assertEquals([
            'lat' => 21.2854,
            'lng' => 39.2376,
        ], $cityCoords);

        // Test state coordinates
        $stateCoords = $service->getStateCoordinates($state->id);
        $this->assertEquals([
            'lat' => 24.7136,
            'lng' => 46.6753,
        ], $stateCoords);

        // Test best coordinates
        $bestCoords = $service->getBestCoordinates($city->id, $state->id);
        $this->assertEquals([
            'lat' => 21.2854,
            'lng' => 39.2376,
        ], $bestCoords);
    }

    public function test_coordinates_migration_structure(): void
    {
        // Verify that the coordinates columns exist in both tables
        $this->assertTrue(\Schema::hasColumn('states', 'lat'));
        $this->assertTrue(\Schema::hasColumn('states', 'lng'));
        $this->assertTrue(\Schema::hasColumn('cities', 'lat'));
        $this->assertTrue(\Schema::hasColumn('cities', 'lng'));
    }

    public function test_state_and_city_models_support_coordinates(): void
    {
        // Test State model with coordinates
        $state = State::create([
            'name' => ['ar' => 'الرياض', 'en' => 'Riyadh'],
            'lat' => 24.7136,
            'lng' => 46.6753,
        ]);

        $this->assertEquals(24.7136, $state->lat);
        $this->assertEquals(46.6753, $state->lng);

        // Test City model with coordinates
        $city = City::create([
            'name' => ['ar' => 'جدة', 'en' => 'Jeddah'],
            'state_id' => $state->id,
            'lat' => 21.2854,
            'lng' => 39.2376,
        ]);

        $this->assertEquals(21.2854, $city->lat);
        $this->assertEquals(39.2376, $city->lng);
        $this->assertEquals($state->id, $city->state_id);
    }
}
