<?php

namespace Tests\Feature;

use App\Models\LicenseRequirement;
use App\Models\User;
use App\Models\WellLicense;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class LicenseRequirementTest extends TestCase
{
    use RefreshDatabase;

    public function test_can_create_license_requirement_for_well_license(): void
    {
        $user = User::factory()->create();
        $wellLicense = WellLicense::factory()->create();

        $requirement = LicenseRequirement::factory()->create([
            'licenseable_id' => $wellLicense->id,
            'licenseable_type' => WellLicense::class,
            'created_by' => $user->id,
        ]);

        $this->assertDatabaseHas('license_requirements', [
            'id' => $requirement->id,
            'licenseable_id' => $wellLicense->id,
            'licenseable_type' => WellLicense::class,
            'status' => LicenseRequirement::STATUS_PENDING,
        ]);
    }

    public function test_can_accept_license_requirement(): void
    {
        $user = User::factory()->create();
        $this->actingAs($user);

        $requirement = LicenseRequirement::factory()->pending()->create();

        $requirement->accept('Looks good');

        $this->assertDatabaseHas('license_requirements', [
            'id' => $requirement->id,
            'status' => LicenseRequirement::STATUS_ACCEPTED,
        ]);
    }

    public function test_can_reject_license_requirement(): void
    {
        $user = User::factory()->create();
        $this->actingAs($user);

        $requirement = LicenseRequirement::factory()->pending()->create();

        $requirement->reject('Not acceptable');

        $this->assertDatabaseHas('license_requirements', [
            'id' => $requirement->id,
            'status' => LicenseRequirement::STATUS_REJECTED,
        ]);
    }

    public function test_can_ask_for_modification(): void
    {
        $user = User::factory()->create();
        $this->actingAs($user);

        $requirement = LicenseRequirement::factory()->pending()->create();

        $requirement->askModification('Please update the document');

        $this->assertDatabaseHas('license_requirements', [
            'id' => $requirement->id,
            'status' => LicenseRequirement::STATUS_ASK_MODIFICATION,
        ]);
    }

    public function test_license_requirement_has_correct_relationship_with_well_license(): void
    {
        $wellLicense = WellLicense::factory()->create();
        $requirement = LicenseRequirement::factory()->create([
            'licenseable_id' => $wellLicense->id,
            'licenseable_type' => WellLicense::class,
        ]);

        $this->assertTrue($requirement->licenseable->is($wellLicense));
        $this->assertTrue($wellLicense->requirements->contains($requirement));
    }
}
