<?php

namespace Tests\Feature;

use App\Filament\Pages\DonorInformationDashboard;
use App\Models\DonorFinancialPortfolio;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Spatie\Permission\Models\Role;
use Tests\TestCase;

class DonorInformationDashboardAuthorizationTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();

        // Create roles
        Role::create(['name' => 'donor_viewer', 'guard_name' => 'web']);
        Role::create(['name' => 'super_admin', 'guard_name' => 'web']);
        Role::create(['name' => 'regular_user', 'guard_name' => 'web']);
    }

    /** @test */
    public function donor_viewer_can_access_dashboard()
    {
        $user = User::factory()->create();
        $user->assignRole('donor_viewer');

        $this->actingAs($user);

        $this->assertTrue(DonorInformationDashboard::canAccess());
    }

    /** @test */
    public function super_admin_can_access_dashboard()
    {
        $user = User::factory()->create();
        $user->assignRole('super_admin');

        $this->actingAs($user);

        $this->assertTrue(DonorInformationDashboard::canAccess());
    }

    /** @test */
    public function regular_user_cannot_access_dashboard()
    {
        $user = User::factory()->create();
        $user->assignRole('regular_user');

        $this->actingAs($user);

        $this->assertFalse(DonorInformationDashboard::canAccess());
    }

    /** @test */
    public function guest_cannot_access_dashboard()
    {
        $this->assertFalse(DonorInformationDashboard::canAccess());
    }

    /** @test */
    public function donor_viewer_only_sees_own_portfolios()
    {
        // Create two users with donor_viewer role
        $user1 = User::factory()->create();
        $user1->assignRole('donor_viewer');

        $user2 = User::factory()->create();
        $user2->assignRole('donor_viewer');

        // Create portfolios for each user
        $this->actingAs($user1);
        $portfolio1 = DonorFinancialPortfolio::factory()->create([
            'name' => 'Portfolio 1',
            'balance' => 10000,
            'created_by' => $user1->id,
        ]);

        $this->actingAs($user2);
        $portfolio2 = DonorFinancialPortfolio::factory()->create([
            'name' => 'Portfolio 2',
            'balance' => 20000,
            'created_by' => $user2->id,
        ]);

        // User 1 should only see their portfolio
        $this->actingAs($user1);
        $user1Portfolios = DonorFinancialPortfolio::all();
        $this->assertCount(1, $user1Portfolios);
        $this->assertEquals($portfolio1->id, $user1Portfolios->first()->id);

        // User 2 should only see their portfolio
        $this->actingAs($user2);
        $user2Portfolios = DonorFinancialPortfolio::all();
        $this->assertCount(1, $user2Portfolios);
        $this->assertEquals($portfolio2->id, $user2Portfolios->first()->id);
    }

    /** @test */
    public function super_admin_sees_all_portfolios()
    {
        // Create regular user and super admin
        $regularUser = User::factory()->create();
        $regularUser->assignRole('donor_viewer');

        $superAdmin = User::factory()->create();
        $superAdmin->assignRole('super_admin');

        // Create portfolio as regular user
        $this->actingAs($regularUser);
        $portfolio = DonorFinancialPortfolio::factory()->create([
            'name' => 'Portfolio 1',
            'balance' => 10000,
            'created_by' => $regularUser->id,
        ]);

        // Super admin should see all portfolios (without scope)
        $this->actingAs($superAdmin);
        $allPortfolios = DonorFinancialPortfolio::withoutAccessScope()->get();
        $this->assertCount(1, $allPortfolios);
        $this->assertEquals($portfolio->id, $allPortfolios->first()->id);
    }
}
