# Old Database Migration Guide

This guide explains how to migrate data from your old database with different schema to the new Laravel application.

## Overview

The migration command transfers:
- **Associations** from the old `association` table to the new `associations` table
- **Users** from the old `users` table to the new `users` table

## Prerequisites

1. **Database Access**: Ensure you have read access to the old database
2. **Environment Configuration**: Configure old database connection in `.env`
3. **Backup**: Always backup your current database before migration
4. **Dependencies**: Ensure PHP PDO extensions are installed for your database type

## Configuration

### 1. Environment Variables

Add these variables to your `.env` file:

```env
# Old Database Connection
OLD_DB_DRIVER=mysql
OLD_DB_HOST=127.0.0.1
OLD_DB_PORT=3306
OLD_DB_DATABASE=your_old_database_name
OLD_DB_USERNAME=your_old_database_username
OLD_DB_PASSWORD=your_old_database_password
```

### 2. Test Connection

Before running the migration, test the connection:

```bash
php artisan migrate:old-database --dry-run
```

## Database Relationships

The old database has a complex structure where association data is spread across multiple related tables:

### Primary Tables Used:
- **`sek_associations`**: Main association table
- **`sek_users`**: User table (association admin details stored here)  
- **`sek_addresses`**: Address information for users
- **`sek_user_bank_information`**: Banking details for users
- **`sek_banks`**: Bank reference data

### Key Relationships:
- Association → Admin User: `sek_associations.association_admin_id = sek_users.id`
- Admin User → Address: `sek_users.id = sek_addresses.user_id`
- Admin User → Bank Info: `sek_users.id = sek_user_bank_information.user_id`
- Bank Info → Bank: `sek_user_bank_information.bank_id = sek_banks.id`

### Special Data Handling:
- **Coordinates**: Stored as comma-separated string (e.g., "24.75099448167933,46.82626031787359")
- **Verification Status**: Derived from `association_admin_verified_at` timestamp presence
- **Missing Data**: Gracefully handled with NULL values or defaults

## Data Mapping

### Association Fields Mapping

| New Database Field | Old Database Source | Notes |
|-------------------|-------------------|--------|
| `name` | `sek_associations.name_en` / `name_ar` | Prioritizes English, falls back to Arabic |
| `license_number` | `sek_associations.license_number` | Direct mapping |
| `license_expiration_date` | `sek_users.license_expiration_date` | From association admin user |
| `representative_full_name` | `sek_users.entered_association_owner_fullname` | From association admin user |
| `representative_position` | `sek_users.position_at_association` | From association admin user |
| `phone_number` | `sek_users.mobile_number` | From association admin user |
| `email` | `sek_users.email` | From association admin user |
| `city` | `sek_addresses.city_ar` | From admin user's address |
| `postal_code` | `sek_addresses.postcode` | From admin user's address |
| `neighborhood` | `sek_addresses.area_ar` | From admin user's address |
| `building_number` | `sek_addresses.building_number` | From admin user's address |
| `street` | `sek_addresses.street_name_ar` | From admin user's address |
| `branch_number` | `sek_addresses.additional_number` | From admin user's address |
| `lat` | Parsed from `sek_addresses.obj_lat_lng` | First value from coordinate string |
| `lng` | Parsed from `sek_addresses.obj_lat_lng` | Second value from coordinate string |
| `iban` | `sek_user_bank_information.iban` | From admin user's bank info |
| `bank_name` | `sek_banks.name_ar` | Via bank_id relationship |
| `account_holder_name` | `sek_user_bank_information.full_name` | From admin user's bank info |
| `is_verified` | `sek_users.association_admin_verified_at != null` | Boolean conversion |
| `is_locked` | `sek_users.is_profile_edit_locked` | Direct mapping |
| `created_at` | `sek_associations.created_at` | Direct mapping |
| `updated_at` | `sek_associations.modified_at` | Maps to updated_at |

### User Fields Mapping

| Old Database Field | New Database Field | Notes |
|-------------------|-------------------|--------|
| `first_name_en + second_name_en + third_name_en + last_name_en` | `name` | Combines all name fields |
| `mobile_number` | `phone_number` | Direct mapping |
| `email` | `email` | Direct mapping with validation |
| `association_id` | `association_id` | Direct mapping |
| `email_is_verified` | `email_verified_at` | Sets timestamp if verified |
| All users | `password` | **Set to default: `TempPassword@123`** |

### Skipped Fields

These fields from the old database are **not migrated** due to schema differences:

#### User Fields Not Migrated:
- `position_at_association`
- `is_association_admin`
- `gender`
- `date_of_birth_gregorian`
- `lang`
- `is_term_accepted`
- `role`
- `user_photo` (requires manual file migration)
- `is_profile_edit_locked`
- `profile_locked_once`
- `is_user_locked`
- `geolocation_update_status`

## Migration Commands

### Dry Run (Recommended First)
Test the migration without inserting data:
```bash
php artisan migrate:old-database --dry-run
```

### Full Migration
Migrate both associations and users:
```bash
php artisan migrate:old-database
```

### Migrate Only Associations
```bash
php artisan migrate:old-database --associations-only
```

### Migrate Only Users
```bash
php artisan migrate:old-database --users-only
```

### Custom Parameters
```bash
# Custom batch size (default: 100)
php artisan migrate:old-database --batch-size=50

# Start from specific record
php artisan migrate:old-database --start-from=1000

# Custom connection name
php artisan migrate:old-database --connection=my_old_db
```

## Post-Migration Steps

### 1. Password Reset
All migrated users have the default password: `TempPassword@123`

**Important**: Ensure users change passwords on first login.

### 2. File Migration
Files are not automatically migrated:
- Association logos (`association_logo`)
- User photos (`user_photo`)

You'll need to manually copy files from the old system to the new media storage.

### 3. Data Verification
After migration, verify:
- User count matches expected numbers
- Associations are properly linked
- Email addresses are valid and unique
- Phone numbers are formatted correctly

### 4. Role Assignment
If your old system had roles/permissions, you'll need to:
1. Identify admin users from the old `is_association_admin` field
2. Assign appropriate roles using Spatie Permission package

## Troubleshooting

### Connection Issues
```
❌ Failed to connect to old database
```
- Verify database credentials in `.env`
- Check network connectivity
- Ensure database user has read permissions

### Validation Errors
```
❌ Error migrating user ID X: validation failed
```
- Check for duplicate email addresses
- Verify email format validity
- Ensure required fields have data

### Memory Issues
For large datasets:
```bash
# Reduce batch size
php artisan migrate:old-database --batch-size=25

# Process in chunks
php artisan migrate:old-database --start-from=0 --batch-size=100
php artisan migrate:old-database --start-from=100 --batch-size=100
```

## Migration Output Example

```
🚀 Starting migration from old database...
🔍 Testing connection to: old_database
✅ Old database connection validated
👥 Starting associations migration...
📊 Found 150 associations to migrate
█████████████████████████████████████████████████ 100% (150/150)
✅ Associations migration completed
👤 Starting users migration...
📊 Found 500 users to migrate
█████████████████████████████████████████████████ 100% (500/500)
✅ Users migration completed

📊 Migration Results:

👥 Associations:
   Total: 150
   Migrated: 148
   Skipped: 2
   Errors: 0

👤 Users:
   Total: 500
   Migrated: 485
   Skipped: 12
   Errors: 3

🔐 All migrated users have the default password: TempPassword@123
⚠️  Please ensure users change their passwords on first login
📁 Files (logos, photos) require manual migration
```

## Security Considerations

1. **Database Access**: Use read-only database user for old database if possible
2. **Password Security**: Default password should be changed immediately
3. **Data Validation**: All migrated data is validated before insertion
4. **Transaction Safety**: Migration runs in database transaction
5. **Logging**: All operations are logged for audit trail

## Best Practices

1. **Always test first**: Use `--dry-run` before actual migration
2. **Backup first**: Backup your new database before migration
3. **Small batches**: Use smaller batch sizes for large datasets
4. **Monitor progress**: Watch for warnings and errors during migration
5. **Verify results**: Check migrated data thoroughly
6. **Document changes**: Keep record of any manual adjustments made

## Support

If you encounter issues:
1. Check the command output for specific error messages
2. Verify database connectivity and permissions
3. Review data validation requirements
4. Check for duplicate or invalid data in source database

For custom field mapping or additional migration requirements, modify the `MigrateOldDatabaseCommand.php` file.
