# Licensing Models Implementation

## Overview
Created two new models for managing well and station licenses, similar to the Order model structure. These models follow Laravel and Filament best practices with proper relationships, media collections, and activity logging.

## Models Created

### 1. WellLicense Model (ترخيص الابار)

**File**: `app/Models/WellLicense.php`

#### Attributes
1. **المنطقة - المحافظة - المركز** (`state_id`, `city_id`, `center_id`)
   - Foreign keys to State, City, and Center models
   - Required fields marked with (**)

2. **اسم البئر** (`well_name`) **
   - Type: `string`
   - Required field

3. **إحداثيات البئر** (`lat`, `lng`) **
   - Type: `decimal(10, 8)` for lat, `decimal(11, 8)` for lng
   - Supports Google Maps integration
   - Required fields

4. **عمق البئر** (`well_depth`)
   - Type: `decimal(8, 2)`
   - Optional field

5. **قطر البئر** (`well_diameter`)
   - Type: `decimal(8, 2)`
   - Optional field

6. **حجم المضخة** (`pump_size`)
   - Type: `decimal(8, 2)`
   - Optional field

7. **كمية السحب جالون/بالدقيقة** (`withdrawal_amount`)
   - Type: `decimal(10, 2)`
   - Optional field

#### Required File Uploads (Media Collections)

1. **survey_plan** (*)
   - رفع مساحي يوضح الموقع العام وموقع البئر
   - Required
   - Accepts: PDF, JPEG, PNG

2. **well_disclosure**
   - إفصاح البئر
   - Optional (إن وجد)
   - Accepts: PDF, JPEG, PNG

3. **water_sample_analysis** (*)
   - تحليل عينة مياه البئر
   - Required
   - Accepts: PDF, JPEG, PNG

4. **ownership_document**
   - صك الملكية او قرار تخصيص او إفادة من امارة المركز بإن الموقع لا يوجد عليها نزاعات قبلية أو فردية
   - Optional (إن وجد)
   - Accepts: PDF, JPEG, PNG

#### Relationships
- `belongsTo(Association::class)` - Associated with an association
- `belongsTo(User::class, 'created_by')` - Creator user
- `belongsTo(State::class)` - State/Region
- `belongsTo(City::class)` - City/Governorate
- `belongsTo(Center::class)` - Center

#### Features
- ✅ Google Maps integration (lat/lng)
- ✅ Activity logging with Spatie Activitylog
- ✅ Media library integration with Spatie MediaLibrary
- ✅ Auto-set created_by on creation
- ✅ Proper type casting for all fields

---

### 2. StationLicense Model (ترخيص المحطات)

**File**: `app/Models/StationLicense.php`

#### Attributes

1. **احداثيات المحطة** (`lat`, `lng`) **
   - Type: `decimal(10, 8)` for lat, `decimal(11, 8)` for lng
   - Supports Google Maps integration
   - Required fields

2. **اسم المحطة** (`station_name`) **
   - Type: `string`
   - Required field

3. **مساحة الموقع** (`site_area`)
   - Type: `decimal(10, 2)`
   - Unit: متر مربع (square meters)
   - Optional field

4. **الطاقة التصميمية** (`design_capacity`)
   - Type: `decimal(10, 2)`
   - Unit: متر مكعب (cubic meters)
   - Optional field

5. **هل يوجد تصريح تشغيلي من المركز الوطني للرقابة على الالتزام البيئي** (`has_environmental_permit`)
   - Type: `boolean`
   - Default: `false`
   - نعم/لا (Yes/No)

6. **هل يوجد أحواض تبخير** (`has_evaporation_ponds`)
   - Type: `boolean`
   - Optional field
   - نعم/لا (Yes/No)
   - Conditional: Only shown if `has_environmental_permit` is `false`

7. **مساحة الاحواض** (`evaporation_ponds_area`)
   - Type: `decimal(10, 2)`
   - Optional field
   - Conditional: Only shown if `has_evaporation_ponds` is `true`

8. **تاريخ بدء تشغيل المحطة** (`operation_start_date`) **
   - Type: `date`
   - Required field

#### Required File Uploads (Media Collections)

1. **survey_plan** (*)
   - رفع مساحي يوضح الموقع العام وموقع البئر
   - Required
   - Accepts: PDF, JPEG, PNG

2. **well_water_analysis** (*)
   - تحليل عينة مياه البئر
   - Required
   - Accepts: PDF, JPEG, PNG

3. **produced_water_analysis** (*)
   - تحليل عينة المياه المنتجة
   - Required
   - Accepts: PDF, JPEG, PNG

4. **reject_water_analysis** (*)
   - تحليل عينة مياه الرجيع
   - Required
   - Accepts: PDF, JPEG, PNG

5. **station_photos**
   - صور (واجهة المحطة-أحواض التبخير-البئر-الخزانات-منطقة التعبئة- الاشياب)
   - Multiple images allowed
   - Accepts: JPEG, PNG

6. **environmental_permit**
   - ارفق التصريح البيئي من المركز الوطني للرقابة على الالتزام البيئي
   - Accepts: PDF, JPEG, PNG

#### Relationships
- `belongsTo(Association::class)` - Associated with an association
- `belongsTo(User::class, 'created_by')` - Creator user

#### Features
- ✅ Google Maps integration (lat/lng)
- ✅ Activity logging with Spatie Activitylog
- ✅ Media library integration with Spatie MediaLibrary
- ✅ Auto-set created_by on creation
- ✅ Proper type casting for all fields
- ✅ Conditional fields logic support

---

## Database Schema

### well_licenses Table
```sql
- id (bigint, primary key)
- association_id (bigint, foreign key)
- state_id (bigint, foreign key, nullable)
- city_id (bigint, foreign key, nullable)
- center_id (bigint, foreign key, nullable)
- well_name (varchar)
- lat (decimal 10,8, nullable)
- lng (decimal 11,8, nullable)
- well_depth (decimal 8,2, nullable)
- well_diameter (decimal 8,2, nullable)
- pump_size (decimal 8,2, nullable)
- withdrawal_amount (decimal 10,2, nullable)
- created_by (bigint, foreign key)
- created_at (timestamp)
- updated_at (timestamp)
```

### station_licenses Table
```sql
- id (bigint, primary key)
- association_id (bigint, foreign key)
- station_name (varchar)
- lat (decimal 10,8, nullable)
- lng (decimal 11,8, nullable)
- site_area (decimal 10,2, nullable)
- design_capacity (decimal 10,2, nullable)
- has_environmental_permit (boolean, default false)
- has_evaporation_ponds (boolean, nullable)
- evaporation_ponds_area (decimal 10,2, nullable)
- operation_start_date (date)
- created_by (bigint, foreign key)
- created_at (timestamp)
- updated_at (timestamp)
```

## Migrations
- `2025_10_02_203115_create_well_licenses_table.php`
- `2025_10_02_203121_create_station_licenses_table.php`

Both migrations have been successfully executed.

## Code Quality
✅ All code formatted with Laravel Pint
✅ No compilation errors
✅ Follows Laravel and Filament best practices
✅ Follows project conventions (PSR-12, type hints, etc.)
✅ Similar structure to Order model
✅ Proper activity logging implemented
✅ Media collections properly configured

## Usage Examples

### Creating a WellLicense
```php
use App\Models\WellLicense;

$wellLicense = WellLicense::create([
    'association_id' => 1,
    'state_id' => 1,
    'city_id' => 1,
    'center_id' => 1,
    'well_name' => 'بئر الخير',
    'lat' => 24.7136,
    'lng' => 46.6753,
    'well_depth' => 150.00,
    'well_diameter' => 12.00,
    'pump_size' => 50.00,
    'withdrawal_amount' => 100.00,
]);

// Add required files
$wellLicense->addMedia($surveyPlan)->toMediaCollection('survey_plan');
$wellLicense->addMedia($waterAnalysis)->toMediaCollection('water_sample_analysis');
```

### Creating a StationLicense
```php
use App\Models\StationLicense;

$stationLicense = StationLicense::create([
    'association_id' => 1,
    'station_name' => 'محطة التنقية',
    'lat' => 24.7136,
    'lng' => 46.6753,
    'site_area' => 500.00,
    'design_capacity' => 1000.00,
    'has_environmental_permit' => false,
    'has_evaporation_ponds' => true,
    'evaporation_ponds_area' => 200.00,
    'operation_start_date' => '2025-01-01',
]);

// Add required files
$stationLicense->addMedia($surveyPlan)->toMediaCollection('survey_plan');
$stationLicense->addMedia($wellWaterAnalysis)->toMediaCollection('well_water_analysis');
$stationLicense->addMedia($producedWaterAnalysis)->toMediaCollection('produced_water_analysis');
$stationLicense->addMedia($rejectWaterAnalysis)->toMediaCollection('reject_water_analysis');
```

### Accessing Location Data
```php
$wellLicense = WellLicense::find(1);

// Get location as array
$location = $wellLicense->location; // ['lat' => 24.7136, 'lng' => 46.6753]

// Set location
$wellLicense->location = ['lat' => 24.5, 'lng' => 46.5];
$wellLicense->save();
```

## Next Steps

The following items still need to be created for these models:

### Resources (Filament Admin Panel)
- [ ] `WellLicenseResource.php`
- [ ] `StationLicenseResource.php`

### Resource Pages
- [ ] `WellLicenseResource/Pages/ListWellLicenses.php`
- [ ] `WellLicenseResource/Pages/CreateWellLicense.php`
- [ ] `WellLicenseResource/Pages/EditWellLicense.php`
- [ ] `WellLicenseResource/Pages/ViewWellLicense.php`
- [ ] `StationLicenseResource/Pages/ListStationLicenses.php`
- [ ] `StationLicenseResource/Pages/CreateStationLicense.php`
- [ ] `StationLicenseResource/Pages/EditStationLicense.php`
- [ ] `StationLicenseResource/Pages/ViewStationLicense.php`

### Translations
- [ ] Arabic translations in `lang/ar/well_license.php`
- [ ] English translations in `lang/en/well_license.php`
- [ ] Arabic translations in `lang/ar/station_license.php`
- [ ] English translations in `lang/en/station_license.php`

### Policies
- [ ] `WellLicensePolicy.php`
- [ ] `StationLicensePolicy.php`

### Tests
- [ ] Feature tests for WellLicense
- [ ] Feature tests for StationLicense

### Factories (Optional)
- [ ] `WellLicenseFactory.php`
- [ ] `StationLicenseFactory.php`

### Seeders (Optional)
- [ ] `WellLicenseSeeder.php`
- [ ] `StationLicenseSeeder.php`

## Notes

1. **Conditional Fields**: The StationLicense model has conditional logic:
   - `has_evaporation_ponds` field should only appear if `has_environmental_permit` is `false`
   - `evaporation_ponds_area` field should only appear if `has_evaporation_ponds` is `true`
   - This logic needs to be implemented in the Filament Resource form

2. **Required Fields**: Fields marked with (**) in the requirements are considered required and should be validated in the Filament forms

3. **File Requirements**: Fields marked with (*) in the file uploads are required files

4. **Similar to Order Model**: Both models follow the same patterns as the Order model:
   - Media library integration
   - Activity logging
   - Google Maps location support
   - Relationships with Association and User
   - Auto-setting created_by
   - Proper type casting

## Files Created
1. `app/Models/WellLicense.php`
2. `app/Models/StationLicense.php`
3. `database/migrations/2025_10_02_203115_create_well_licenses_table.php`
4. `database/migrations/2025_10_02_203121_create_station_licenses_table.php`
5. `docs/LICENSING_MODELS.md` (this file)
