# Association Type Field Implementation

## Overview
Added a new `type` field to the Association model that allows choosing between three organization types:
- جمعية (Association)
- مؤسسة (Institution)
- وقف (Endowment)

The default value is 'جمعية' (Association).

## Implementation Details

### 1. Enum Created
**File**: `app/Enums/AssociationType.php`

A Filament-compatible enum implementing `HasColor` and `HasLabel` interfaces:
- `ASSOCIATION` → 'association' (Green badge, default)
- `INSTITUTION` → 'institution' (Blue badge)
- `ENDOWMENT` → 'endowment' (Yellow badge)

Each case provides:
- Translated labels via `getLabel()` method
- Color coding for Filament badges via `getColor()` method
- `getOptions()` static method for form dropdowns

### 2. Database Migrations
**Files**:
- `database/migrations/2025_10_02_201327_add_type_to_associations_table.php`
- `database/migrations/2025_10_02_201616_add_type_to_association_registrations_table.php`

Added `type` column to both `associations` and `association_registrations` tables:
- Type: `string`
- Default: `'association'`
- Position: After `name` column

### 3. Model Updates

#### Association Model (`app/Models/Association.php`)
- Added `type` to `$fillable` array
- Added `type => AssociationType::class` to `$casts` array
- Imported `AssociationType` enum

#### AssociationRegistration Model (`app/Models/AssociationRegistration.php`)
- Added `type` to `$fillable` array
- Added `type => AssociationType::class` to `$casts` array
- Imported `AssociationType` enum

### 4. Filament Resource Updates

#### AssociationResource (`app/Filament/Resources/AssociationResource.php`)
- Added `Select` component for type field in form:
  - Uses `AssociationType::getOptions()` for options
  - Default value: `AssociationType::ASSOCIATION->value`
  - Non-native select for better UX
  - Positioned after the `name` field
- Added type column to table:
  - Displays as colored badge
  - Sortable and toggleable
  - Positioned after the `name` column

### 5. Registration Form Updates

#### RegisterAssociation Livewire Component (`app/Livewire/RegisterAssociation.php`)
- Added `Select` component for type field
- Included `type` in form state management
- Updated association creation to include type field
- Added fallback to default value if not provided

### 6. Translations

#### Arabic (`lang/ar/association.php`)
```php
'type_label' => 'النوع',
'type' => [
    'association' => 'جمعية',
    'institution' => 'مؤسسة',
    'endowment' => 'وقف',
],
```

#### English (`lang/en/association.php`)
```php
'type_label' => 'Type',
'type' => [
    'association' => 'Association',
    'institution' => 'Institution',
    'endowment' => 'Endowment',
],
```

### 7. Testing
**File**: `tests/Feature/AssociationTypeTest.php`

Comprehensive test suite covering:
- Creating associations with different types
- Default type behavior
- Enum label translations
- Enum color assignments
- Options array generation

## Usage Examples

### Creating an Association with Type
```php
use App\Models\Association;
use App\Enums\AssociationType;

$association = Association::create([
    'name' => 'مؤسسة الخير',
    'type' => AssociationType::INSTITUTION,
    // ... other fields
]);
```

### Accessing Type Information
```php
$association = Association::find(1);

// Get enum instance
$type = $association->type; // AssociationType enum

// Get value
$value = $association->type->value; // 'association', 'institution', or 'endowment'

// Get translated label
$label = $association->type->getLabel(); // 'جمعية', 'مؤسسة', or 'وقف'

// Get badge color
$color = $association->type->getColor(); // 'success', 'info', or 'warning'
```

### Using in Forms
```php
Forms\Components\Select::make('type')
    ->label(__('association.type_label'))
    ->options(AssociationType::getOptions())
    ->default(AssociationType::ASSOCIATION->value)
    ->required()
    ->native(false)
```

## User Interface

### Admin Panel (AssociationResource)
- Form: Dropdown select field with three options
- Table: Colored badge showing type with appropriate color coding
  - Green for جمعية (Association)
  - Blue for مؤسسة (Institution)
  - Yellow for وقف (Endowment)

### Public Registration (RegisterAssociation)
- Form: Dropdown select field with three options
- Default selection: جمعية (Association)
- Field positioned after association name

## Migration Status
✅ Both migrations have been successfully executed
✅ Type field added to `associations` table
✅ Type field added to `association_registrations` table

## Code Quality
✅ All code formatted with Laravel Pint
✅ No compilation errors
✅ Follows Laravel and Filament best practices
✅ Follows project conventions (PSR-12, type hints, etc.)

## Files Modified
1. `app/Enums/AssociationType.php` (new)
2. `app/Models/Association.php`
3. `app/Models/AssociationRegistration.php`
4. `app/Filament/Resources/AssociationResource.php`
5. `app/Livewire/RegisterAssociation.php`
6. `lang/ar/association.php`
7. `lang/en/association.php`
8. `database/migrations/2025_10_02_201327_add_type_to_associations_table.php` (new)
9. `database/migrations/2025_10_02_201616_add_type_to_association_registrations_table.php` (new)
10. `tests/Feature/AssociationTypeTest.php` (new)
