<?php

use App\Models\Order;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Add WELL_PURIFICATION_LICENSES type to the enum
        $types = [
            'STATION',
            'TANKER',
            'STORAGE_TANK',
            'CENTRAL_STORAGE_TANK',
            'MAINTENANCE_STATION',
            'MAINTENANCE_TANKER',
            'MICRO_NETWORK',
            'OTHER',
            'WELL_DRILLING',
            'RAINWATER_STORAGE',
            'HOME_FILTERS',
            'WATER_BOTTLES',
            'COOLERS',
            'PRODUCTS_UNDER_REVIEW',
            'WELL_PURIFICATION_LICENSES',
        ];

        $this->updateOrderTypeEnum($types);
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // First, update any records with the new type back to OTHER
        DB::statement("UPDATE orders SET type = 'OTHER' WHERE type = 'WELL_PURIFICATION_LICENSES'");

        // Remove WELL_PURIFICATION_LICENSES type from the enum
        $types = [
            'STATION',
            'TANKER',
            'STORAGE_TANK',
            'CENTRAL_STORAGE_TANK',
            'MAINTENANCE_STATION',
            'MAINTENANCE_TANKER',
            'MICRO_NETWORK',
            'OTHER',
            'WELL_DRILLING',
            'RAINWATER_STORAGE',
            'HOME_FILTERS',
            'WATER_BOTTLES',
            'COOLERS',
            'PRODUCTS_UNDER_REVIEW',
        ];

        $this->updateOrderTypeEnum($types);
    }

    /**
     * Update the order type enum for different database drivers
     */
    private function updateOrderTypeEnum(array $types): void
    {
        $driver = DB::connection()->getDriverName();
        $enumValues = "'".implode("','", $types)."'";
        $defaultValue = 'STATION';

        switch ($driver) {
            case 'mysql':
                DB::statement("ALTER TABLE orders MODIFY COLUMN type ENUM($enumValues) NOT NULL DEFAULT '$defaultValue'");
                break;

            case 'pgsql':
                DB::statement('ALTER TABLE orders DROP CONSTRAINT IF EXISTS orders_type_check');

                DB::statement("ALTER TABLE orders 
                    ADD CONSTRAINT orders_type_check CHECK (type IN ($enumValues))");
                break;

            case 'sqlite':
                // SQLite doesn't have native ENUM support
                // We'll skip ENUM constraints for SQLite and rely on application-level validation
                Log::info('SQLite detected. ENUM constraints not applied - relying on application validation.');
                break;

            default:
                // Fallback for other databases - attempt MySQL syntax but warn
                Log::warning("Unsupported database driver: $driver. Attempting MySQL syntax.");
                DB::statement("ALTER TABLE orders MODIFY COLUMN type ENUM($enumValues) NOT NULL DEFAULT '$defaultValue'");
                break;
        }
    }
};
