<?php

use App\Models\ProjectCharterChangeRequest;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Add first_approved status to the enum
        $statuses = [
            ProjectCharterChangeRequest::STATUS_PENDING,
            ProjectCharterChangeRequest::STATUS_FIRST_APPROVED,
            ProjectCharterChangeRequest::STATUS_APPROVED,
            ProjectCharterChangeRequest::STATUS_REJECTED,
        ];

        $this->updateProjectCharterChangeRequestStatusEnum($statuses);
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // First, update any records with the new status back to pending
        DB::statement("UPDATE project_charter_change_requests SET status = 'pending' WHERE status = 'first_approved'");

        // Remove first_approved status from the enum
        $statuses = [
            ProjectCharterChangeRequest::STATUS_PENDING,
            ProjectCharterChangeRequest::STATUS_APPROVED,
            ProjectCharterChangeRequest::STATUS_REJECTED,
        ];

        $this->updateProjectCharterChangeRequestStatusEnum($statuses);
    }

    /**
     * Update the project charter change request status enum for different database drivers
     */
    private function updateProjectCharterChangeRequestStatusEnum(array $statuses): void
    {
        $driver = DB::connection()->getDriverName();
        $enumValues = "'".implode("','", $statuses)."'";
        $defaultValue = ProjectCharterChangeRequest::STATUS_PENDING;

        switch ($driver) {
            case 'mysql':
                DB::statement("ALTER TABLE project_charter_change_requests MODIFY COLUMN status ENUM($enumValues) NOT NULL DEFAULT '$defaultValue'");
                break;

            case 'pgsql':
                DB::statement('ALTER TABLE project_charter_change_requests DROP CONSTRAINT IF EXISTS project_charter_change_requests_status_check');

                DB::statement("ALTER TABLE project_charter_change_requests 
                    ADD CONSTRAINT project_charter_change_requests_status_check CHECK (status IN ($enumValues))");
                break;

            case 'sqlite':
                // SQLite doesn't have native ENUM support
                // We'll skip ENUM constraints for SQLite and rely on application-level validation
                Log::info('SQLite detected. ENUM constraints not applied - relying on application validation.');
                break;

            default:
                // Fallback for other databases - attempt MySQL syntax but warn
                Log::warning("Unsupported database driver: $driver. Attempting MySQL syntax.");
                DB::statement("ALTER TABLE project_charter_change_requests MODIFY COLUMN status ENUM($enumValues) NOT NULL DEFAULT '$defaultValue'");
                break;
        }
    }
};
