<?php

use App\Models\ExchangeRequest;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Add TRANSACTION_APPROVED status to the enum
        $statuses = [
            ExchangeRequest::STATUS_CREATED,
            ExchangeRequest::STATUS_FINANCIAL_APPROVED,
            ExchangeRequest::STATUS_SHARED_SERVICES_APPROVED,
            ExchangeRequest::STATUS_CEO_APPROVED,
            ExchangeRequest::STATUS_TRANSACTION_APPROVED,
            ExchangeRequest::STATUS_APPROVED,
            ExchangeRequest::STATUS_REJECTED,
        ];

        $this->updateExchangeRequestStatusEnum($statuses);
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // First, update any records with the new status back to ceo_approved
        DB::statement("UPDATE exchange_requests SET status = 'ceo_approved' WHERE status = 'transaction_approved'");

        // Remove TRANSACTION_APPROVED status from the enum
        $statuses = [
            ExchangeRequest::STATUS_CREATED,
            ExchangeRequest::STATUS_FINANCIAL_APPROVED,
            ExchangeRequest::STATUS_SHARED_SERVICES_APPROVED,
            ExchangeRequest::STATUS_CEO_APPROVED,
            ExchangeRequest::STATUS_APPROVED,
            ExchangeRequest::STATUS_REJECTED,
        ];

        $this->updateExchangeRequestStatusEnum($statuses);
    }

    /**
     * Update the exchange request status enum for different database drivers
     */
    private function updateExchangeRequestStatusEnum(array $statuses): void
    {
        $driver = DB::connection()->getDriverName();
        $enumValues = "'".implode("','", $statuses)."'";
        $defaultValue = ExchangeRequest::STATUS_CREATED;

        switch ($driver) {
            case 'mysql':
                DB::statement("ALTER TABLE exchange_requests MODIFY COLUMN status ENUM($enumValues) NOT NULL DEFAULT '$defaultValue'");
                break;

            case 'pgsql':
                DB::statement('ALTER TABLE exchange_requests DROP CONSTRAINT IF EXISTS exchange_requests_status_check');

                DB::statement("ALTER TABLE exchange_requests 
                    ADD CONSTRAINT exchange_requests_status_check CHECK (status IN ($enumValues))");
                break;

            case 'sqlite':
                // SQLite doesn't have native ENUM support
                // We'll skip ENUM constraints for SQLite and rely on application-level validation
                Log::info('SQLite detected. ENUM constraints not applied - relying on application validation.');
                break;

            default:
                // Fallback for other databases - attempt MySQL syntax but warn
                Log::warning("Unsupported database driver: $driver. Attempting MySQL syntax.");
                DB::statement("ALTER TABLE exchange_requests MODIFY COLUMN status ENUM($enumValues) NOT NULL DEFAULT '$defaultValue'");
                break;
        }
    }
};
