<?php

use App\Models\Order;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Add ACCEPTED status to the enum
        $statuses = [
            Order::STATUS_CREATED,
            Order::STATUS_PENDING_RECOMMENDATION,
            Order::STATUS_RECOMMENDED,
            Order::STATUS_TECHNICAL_FINANCIAL_REQUESTED,
            Order::STATUS_TECHNICAL_FINANCIAL_APPROVED,
            Order::STATUS_PROJECT_MANAGER_APPROVED,
            Order::STATUS_CLASSIFICATION,
            Order::STATUS_AGREEMENT,
            Order::STATUS_CEO_APPROVED,
            Order::STATUS_LAUNCHED,
            Order::STATUS_ACCEPTED,
            Order::STATUS_REJECTED,
        ];

        $this->updateOrderStatusEnum($statuses);
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Remove ACCEPTED status from the enum
        $statuses = [
            Order::STATUS_CREATED,
            Order::STATUS_PENDING_RECOMMENDATION,
            Order::STATUS_RECOMMENDED,
            Order::STATUS_TECHNICAL_FINANCIAL_REQUESTED,
            Order::STATUS_TECHNICAL_FINANCIAL_APPROVED,
            Order::STATUS_PROJECT_MANAGER_APPROVED,
            Order::STATUS_CLASSIFICATION,
            Order::STATUS_AGREEMENT,
            Order::STATUS_CEO_APPROVED,
            Order::STATUS_LAUNCHED,
            Order::STATUS_REJECTED,
        ];

        $this->updateOrderStatusEnum($statuses);
    }

    /**
     * Update the order status enum for different database drivers
     */
    private function updateOrderStatusEnum(array $statuses): void
    {
        $driver = DB::connection()->getDriverName();
        $enumValues = "'".implode("','", $statuses)."'";
        $defaultValue = Order::STATUS_PENDING_RECOMMENDATION;

        switch ($driver) {
            case 'mysql':
                DB::statement("ALTER TABLE orders MODIFY COLUMN status ENUM($enumValues) NOT NULL DEFAULT '$defaultValue'");
                break;

            case 'pgsql':
                DB::statement('ALTER TABLE orders DROP CONSTRAINT IF EXISTS orders_status_check');

                DB::statement("ALTER TABLE orders 
                    ADD CONSTRAINT orders_status_check CHECK (status IN ($enumValues))");
                break;

            case 'sqlite':
                // SQLite doesn't have native ENUM support
                // We'll skip ENUM constraints for SQLite and rely on application-level validation
                Log::info('SQLite detected. ENUM constraints not applied - relying on application validation.');
                break;

            default:
                // Fallback for other databases - attempt MySQL syntax but warn
                Log::warning("Unsupported database driver: $driver. Attempting MySQL syntax.");
                DB::statement("ALTER TABLE orders MODIFY COLUMN status ENUM($enumValues) NOT NULL DEFAULT '$defaultValue'");
                break;
        }
    }
};
