<?php

namespace Database\Factories;

use App\Models\Initiative;
use App\Models\InitiativeOperationGoal;
use Illuminate\Database\Eloquent\Factories\Factory;

class InitiativeOperationGoalFactory extends Factory
{
    protected $model = InitiativeOperationGoal::class;

    public function definition(): array
    {
        $goalTypes = [
            'Water Quality Improvement',
            'Infrastructure Efficiency',
            'Community Engagement',
            'Cost Reduction',
            'Service Coverage Expansion',
            'Maintenance Optimization',
            'Emergency Response',
            'Environmental Impact Reduction',
        ];

        $goalType = $this->faker->randomElement($goalTypes);

        return [
            'initiative_id' => Initiative::factory(),
            'name' => $goalType.' Goal',
            'performance_criteria' => $this->generatePerformanceCriteria($goalType),
            'target' => $this->generateTarget($goalType),
            'actual' => $this->faker->boolean(60) ? $this->generateActual($goalType) : null,
        ];
    }

    private function generatePerformanceCriteria(string $goalType): string
    {
        $criteria = [
            'Water Quality Improvement' => 'Water purity levels, contamination reduction percentages, and compliance with health standards',
            'Infrastructure Efficiency' => 'System uptime, maintenance frequency, and operational cost per unit',
            'Community Engagement' => 'Community participation rates, feedback scores, and training completion rates',
            'Cost Reduction' => 'Cost per beneficiary, operational expense ratio, and budget variance',
            'Service Coverage Expansion' => 'Number of new beneficiaries, geographic coverage area, and service accessibility',
            'Maintenance Optimization' => 'Mean time between failures, preventive maintenance completion, and repair response time',
            'Emergency Response' => 'Response time to emergencies, system recovery time, and backup system availability',
            'Environmental Impact Reduction' => 'Carbon footprint reduction, waste minimization, and sustainable resource usage',
        ];

        return $criteria[$goalType] ?? $this->faker->paragraph();
    }

    private function generateTarget(string $goalType): string
    {
        $targets = [
            'Water Quality Improvement' => 'Achieve 99.5% water purity compliance within 6 months',
            'Infrastructure Efficiency' => 'Maintain 98% system uptime with 15% cost reduction',
            'Community Engagement' => 'Train 200+ community members with 85% satisfaction rate',
            'Cost Reduction' => 'Reduce operational costs by 20% while maintaining service quality',
            'Service Coverage Expansion' => 'Extend service to 500+ additional families in target areas',
            'Maintenance Optimization' => 'Reduce emergency repairs by 30% through preventive maintenance',
            'Emergency Response' => 'Achieve <2 hour response time for all water emergencies',
            'Environmental Impact Reduction' => 'Reduce carbon footprint by 25% and achieve zero waste goals',
        ];

        return $targets[$goalType] ?? $this->faker->sentence(10);
    }

    private function generateActual(string $goalType): string
    {
        $actuals = [
            'Water Quality Improvement' => $this->faker->randomElement([
                'Achieved 99.2% compliance - slightly below target',
                'Exceeded target with 99.7% compliance rate',
                'Currently at 97.8% - improvement needed',
            ]),
            'Infrastructure Efficiency' => $this->faker->randomElement([
                'Achieved 97.5% uptime with 18% cost reduction',
                'Exceeded expectations - 99.1% uptime, 22% cost reduction',
                'Below target - 95.2% uptime, 12% cost reduction',
            ]),
            'Community Engagement' => $this->faker->randomElement([
                'Trained 185 members with 88% satisfaction',
                'Exceeded target - 225 members trained, 92% satisfaction',
                'Partially achieved - 156 members, 81% satisfaction',
            ]),
        ];

        return $actuals[$goalType] ?? $this->faker->sentence(8);
    }

    public function completed(): static
    {
        return $this->state(fn (array $attributes) => [
            'actual' => $this->generateActual($this->faker->randomElement([
                'Water Quality Improvement',
                'Infrastructure Efficiency',
                'Community Engagement',
            ])),
        ]);
    }

    public function pending(): static
    {
        return $this->state(fn (array $attributes) => [
            'actual' => null,
        ]);
    }
}
