<?php

namespace App\Services;

use App\Models\City;
use App\Models\State;
use Illuminate\Support\Facades\Log;

class LocationCoordinatesService
{
    /**
     * Get coordinates for a given state ID
     */
    public function getStateCoordinates(int $stateId): ?array
    {
        $state = State::find($stateId);

        if (! $state || ! $state->lat || ! $state->lng) {
            Log::warning("No coordinates found for state ID: {$stateId}");

            return null;
        }

        return [
            'lat' => (float) $state->lat,
            'lng' => (float) $state->lng,
        ];
    }

    /**
     * Get coordinates for a given city ID
     */
    public function getCityCoordinates(int $cityId): ?array
    {
        $city = City::with('state')->find($cityId);

        if (! $city) {
            Log::warning("City not found for ID: {$cityId}");

            return null;
        }

        // Prefer city coordinates if available
        if ($city->lat && $city->lng) {
            return [
                'lat' => (float) $city->lat,
                'lng' => (float) $city->lng,
            ];
        }

        // Fallback to state coordinates if city coordinates are not available
        if ($city->state && $city->state->lat && $city->state->lng) {
            Log::info("Using state coordinates for city ID: {$cityId}");

            return [
                'lat' => (float) $city->state->lat,
                'lng' => (float) $city->state->lng,
            ];
        }

        Log::warning("No coordinates found for city ID: {$cityId}");

        return null;
    }

    /**
     * Get the best available coordinates for a city, with fallback options
     */
    public function getBestCoordinates(?int $cityId = null, ?int $stateId = null): array
    {
        // Default coordinates (Riyadh, Saudi Arabia)
        $defaultCoordinates = [
            'lat' => 24.7136,
            'lng' => 46.6753,
        ];

        // Try city coordinates first
        if ($cityId) {
            $cityCoords = $this->getCityCoordinates($cityId);
            if ($cityCoords) {
                return $cityCoords;
            }
        }

        // Fallback to state coordinates
        if ($stateId) {
            $stateCoords = $this->getStateCoordinates($stateId);
            if ($stateCoords) {
                return $stateCoords;
            }
        }

        // Return default coordinates if nothing else is available
        return $defaultCoordinates;
    }

    /**
     * Get center point coordinates for multiple cities
     */
    public function getCenterCoordinatesForCities(array $cityIds): array
    {
        if (empty($cityIds)) {
            return $this->getBestCoordinates();
        }

        $coordinates = [];
        foreach ($cityIds as $cityId) {
            $coords = $this->getCityCoordinates($cityId);
            if ($coords) {
                $coordinates[] = $coords;
            }
        }

        if (empty($coordinates)) {
            return $this->getBestCoordinates();
        }

        // Calculate center point
        $totalLat = array_sum(array_column($coordinates, 'lat'));
        $totalLng = array_sum(array_column($coordinates, 'lng'));
        $count = count($coordinates);

        return [
            'lat' => $totalLat / $count,
            'lng' => $totalLng / $count,
        ];
    }
}
