<?php

namespace App\Notifications;

use App\Channels\UnifonicSmsChannel;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class SendOtpNotification extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public string $otpCode,
        public string $channel = 'email'
    ) {
        //
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        if ($this->channel === 'sms') {
            return [UnifonicSmsChannel::class];
        }

        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject(__('auth.otp_subject').': '.$this->otpCode)
            ->greeting(__('auth.otp_greeting'))
            ->line(__('auth.otp_message'))
            ->line(__('auth.otp_code').': '.$this->otpCode)
            ->line(__('auth.otp_expires'))
            ->salutation(__('auth.otp_salutation'));
    }

    /**
     * Get the SMS representation of the notification.
     */
    public function toSms(object $notifiable): array
    {
        $phone = $notifiable->phone_number ?? $notifiable->phone ?? null;

        if (! $phone) {
            throw new \Exception('Phone number not found for SMS notification');
        }

        $message = __('auth.sms_otp_message', ['code' => $this->otpCode]);

        return [
            'phone' => $phone,
            'message' => $message,
        ];
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'otp_code' => $this->otpCode,
            'channel' => $this->channel,
        ];
    }
}
