<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Archilex\AdvancedTables\Concerns\HasViews;
use Filament\Models\Contracts\FilamentUser;
use Filament\Panel;
use Heloufir\FilamentKanban\Interfaces\KanbanResourceModel;
use Heloufir\FilamentKanban\ValueObjects\KanbanResource;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Auth;
use Monzer\FilamentWorkflows\Traits\TrackWorkflowModelEvents;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable implements FilamentUser, KanbanResourceModel
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, HasRoles, HasViews, Notifiable, TrackWorkflowModelEvents;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'job_title',
        'email',
        'password',
        'phone_number',
        'otp',
        'otp_expires_at',
        'terms_and_conditions_accepted',
        'association_id',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'otp_expires_at' => 'datetime',
            'terms_and_conditions_accepted' => 'boolean',
            'password' => 'hashed',
        ];
    }

    public function association(): BelongsTo
    {
        return $this->belongsTo(Association::class);
    }

    public function departments(): BelongsToMany
    {
        return $this->belongsToMany(Department::class);
    }

    /**
     * Get the donor financial portfolios this user has access to
     */
    public function donorFinancialPortfolios(): BelongsToMany
    {
        return $this->belongsToMany(DonorFinancialPortfolio::class, 'user_donor_financial_portfolio')
            ->withTimestamps();
    }

    /**
     * Get tasks assigned to this user
     */
    public function assignedTasks(): BelongsToMany
    {
        return $this->belongsToMany(AdminsTask::class, 'admins_task_user')
            ->withTimestamps();
    }

    /**
     * Get tasks created by this user
     */
    public function createdTasks(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(AdminsTask::class, 'created_by');
    }

    public function canAccessPanel(Panel $panel): bool
    {
        // TODO: check later
        return true;
    }

    public function canImpersonate()
    {
        return Auth::user()?->can('can_impersonate_user');
    }

    public function setEmailAttribute($value)
    {
        $this->attributes['email'] = strtolower($value);
    }

    public function getEmailAttribute($value)
    {
        return strtolower($value);
    }

    public function toKanbanResource(): KanbanResource
    {
        return KanbanResource::make()
            ->id($this->id)
            ->name($this->name)
            ->avatar('https://ui-avatars.com/api/?name='.$this->name.'&color=FFFFFF&background=09090b');
    }
}
