<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;

class InitiativeProject extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'weight',
        'initiative_id',
        'created_by',
    ];

    protected $casts = [
        'weight' => 'float',
    ];

    public function initiative(): BelongsTo
    {
        return $this->belongsTo(Initiative::class);
    }

    public function projectCharter(): HasOne
    {
        return $this->hasOne(ProjectCharter::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Boot the model and add validation
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (Auth::check()) {
                $model->created_by = Auth::id();
            }
        });

        static::saving(function ($project) {
            $project->validateWeightLimit();
        });
    }

    /**
     * Validate that total weight doesn't exceed 100% and weight is positive
     */
    protected function validateWeightLimit()
    {
        if (! $this->initiative_id || ! $this->weight) {
            return;
        }

        // Check for zero or negative weight
        if ($this->weight <= 0) {
            throw ValidationException::withMessages([
                'weight' => __('initiative_project.weight_invalid_validation_error'),
            ]);
        }

        // Calculate current total excluding this record if it's being updated
        $query = $this->initiative->projects();
        if ($this->exists) {
            $query->where('id', '!=', $this->id);
        }

        $currentTotal = $query->sum('weight');
        $newTotal = $currentTotal + $this->weight;

        if ($newTotal > 100) {
            $available = 100 - $currentTotal;
            throw ValidationException::withMessages([
                'weight' => __('initiative_project.weight_validation_error', [
                    'available' => $available,
                    'current' => $currentTotal,
                ]),
            ]);
        }
    }
}
