<?php

namespace App\Filament\Widgets;

use App\Models\AdminsTask;
use BezhanSalleh\FilamentShield\Traits\HasWidgetShield;
use Leandrocfe\FilamentApexCharts\Widgets\ApexChartWidget;

class TaskGanttChart extends ApexChartWidget
{
    use HasWidgetShield;

    /**
     * Chart Id
     */
    protected static ?string $chartId = 'taskGanttChart';

    /**
     * Widget Title
     */
    protected static ?string $heading = null;

    public ?string $newHeading = null;

    protected int|string|array $columnSpan = 'full';

    public string $taskModel = AdminsTask::class;

    public ?int $projectCharterId = null;

    public function getHeading(): string
    {
        return $this->newHeading ?? __('widgets.view_admin_tasks_chart');
    }

    /**
     * Chart options (series, labels, types, size, animations...)
     * https://apexcharts.com/docs/options
     */
    protected function getOptions(): array
    {
        $model = $this->taskModel;

        $query = $model::query()
            ->select('name', 'start_date', 'end_date');

        if ($this->projectCharterId) {
            $query->whereHas(
                'deliverable',
                fn ($q) => $q->where('project_charter_id', $this->projectCharterId)
            );
        }

        // sort the query by start_date
        $query->orderBy('start_date');

        $tasks = $query->get();

        if ($tasks->isEmpty()) {
            $tasksData = [[
                'x' => __('widgets.no_tasks'),
                'y' => [now()->timestamp * 1000, now()->timestamp * 1000],
            ]];

            $minDate = $maxDate = now()->timestamp * 1000;
        } else {
            $tasksData = $tasks->map(fn ($task) => [
                'x' => $task->name,
                'y' => [
                    strtotime($task->start_date) * 1000,
                    strtotime($task->end_date) * 1000,
                ],
            ])->toArray();

            $minDate = strtotime($tasks->min('start_date')) * 1000;
            $maxDate = strtotime($tasks->max('end_date')) * 1000;
        }

        return [
            'chart' => [
                'type' => 'rangeBar',
                'height' => 600,
                'fontFamily' => 'inherit',
            ],
            'plotOptions' => [
                'bar' => [
                    'horizontal' => true,
                    'distributed' => true,
                    'isDumbbell' => true,
                    'dumbbellColors' => [['#EC7D31', '#36BDCB']],
                ],
            ],
            'grid' => [
                'show' => true,
                'borderColor' => '#e0e6ed',
                'strokeDashArray' => 0,
                'position' => 'back',
                'xaxis' => [
                    'lines' => [
                        'show' => true,
                    ],
                ],
                'yaxis' => [
                    'lines' => [
                        'show' => true,
                    ],
                ],
                'row' => [
                    'colors' => ['transparent', 'transparent'],
                    'opacity' => 0.5,
                ],
                'column' => [
                    'colors' => ['transparent', 'transparent'],
                    'opacity' => 0.5,
                ],
            ],
            'xaxis' => [
                'type' => 'datetime',
                'min' => $minDate,
                'max' => $maxDate,
            ],
            'series' => [
                [
                    'data' => $tasksData,
                ],
            ],
            'fill' => [
                'type' => 'gradient',
                'gradient' => [
                    'gradientToColors' => ['#36BDCB'],
                    'inverseColors' => false,
                    'stops' => [0, 100],
                ],
            ],
        ];
    }
}
