<?php

namespace App\Filament\Widgets;

use App\Models\ProjectCharter;
use BezhanSalleh\FilamentShield\Traits\HasWidgetShield;
use Filament\Forms\Components\Select;
use Filament\Widgets\Concerns\InteractsWithPageFilters;
use Illuminate\Support\Facades\Log;
use Leandrocfe\FilamentApexCharts\Widgets\ApexChartWidget;

class ProjectProgressGauge extends ApexChartWidget
{
    use HasWidgetShield;
    use InteractsWithPageFilters;

    protected static ?string $chartId = 'projectProgressGauge';

    protected static bool $isLazy = false;

    protected static ?string $pollingInterval = null;

    protected static ?string $heading = null;

    // protected static ?int $sort = 3;

    // protected int|string|array $columnSpan = ['md' => 4, 'xl' => 3];

    public ?int $project_charter_id = null;

    protected function getFormSchema(): array
    {
        return [
            Select::make('project_charter_id')
                ->label(__('widgets.select_project'))
                ->placeholder(__('widgets.select_project_placeholder'))
                ->searchable()
                ->options(fn () => $this->getProjectsOptions())
                ->reactive()
                ->afterStateUpdated(function ($state) {
                    // Keep the Livewire public property in sync so getOptions() uses latest selection
                    $this->project_charter_id = $state ?: null;
                }),
        ];
    }

    public function getHeading(): string
    {
        return __('pages.projects_dashboard').' - '.__('widgets.project_progress_gauge');
    }

    protected function getOptions(): array
    {
        $progress = 0;
        $name = '-';

        if ($this->project_charter_id) {
            $charter = ProjectCharter::with('deliverables')
                ->find($this->project_charter_id);
            if ($charter) {
                $progress = round($charter->project_progress, 2);
                $name = $charter->name;
            }
        } else {
            $first = ProjectCharter::query()
                ->when($this->filters['initiative_id'] ?? null, function ($q, $initiativeId) {
                    $q->whereHas('initiativeProject', function ($sub) use ($initiativeId) {
                        $sub->where('initiative_id', $initiativeId);
                    });
                })
                ->orderBy('id')
                ->first();
            if ($first) {
                $progress = round($first->project_progress, 2);
                $name = $first->name;
            }
        }

        Log::info('Project Progress Gauge for '.$name.': '.$progress.'%');

        return [
            'chart' => [
                'type' => 'radialBar',
                'height' => 320,
                'fontFamily' => 'inherit',
            ],
            'series' => [$progress],
            'labels' => [$name],
            'plotOptions' => [
                'radialBar' => [
                    'startAngle' => -140,
                    'endAngle' => 140,
                    'hollow' => [
                        'size' => '60%',
                    ],
                    'track' => [
                        'background' => '#F1F5F9',
                        'strokeWidth' => '100%',
                    ],
                    'dataLabels' => [
                        'name' => [
                            'fontSize' => '14px',
                            'fontWeight' => 600,
                        ],
                        'value' => [
                            'fontSize' => '28px',
                            'fontWeight' => 700,
                            // 'formatter' => "function (val) { return val + '%'; }",
                        ],
                    ],
                ],
            ],
            'fill' => [
                'type' => 'gradient',
                'gradient' => [
                    'shade' => 'light',
                    'type' => 'horizontal',
                    'shadeIntensity' => 0.4,
                    'gradientToColors' => ['#10B981'],
                    'inverseColors' => false,
                    'opacityFrom' => 0.9,
                    'opacityTo' => 0.9,
                    'stops' => [0, 50, 100],
                ],
            ],
            'colors' => [$this->colorForProgress($progress)],
            'stroke' => [
                'dashArray' => 4,
            ],
            'tooltip' => [
                'enabled' => true,
                'y' => [
                    // 'formatter' => "function (val) { return val + '%'; }",
                ],
            ],
        ];
    }

    private function colorForProgress(float $progress): string
    {
        return $progress >= 80 ? '#10B981' : ($progress >= 50 ? '#F59E0B' : '#EF4444');
    }

    private function getProjectsOptions(): array
    {
        $query = ProjectCharter::query();

        // Apply year filter
        if ($selectedYear = $this->filters['year'] ?? null) {
            $query->where(function ($subQuery) use ($selectedYear) {
                $subQuery->whereYear('start_date', $selectedYear)
                    ->orWhereYear('expected_end_date', $selectedYear);
            });
        }

        if ($initiativeId = $this->filters['initiative_id'] ?? null) {
            $query->whereHas('initiativeProject', function ($q) use ($initiativeId) {
                $q->where('initiative_id', $initiativeId);
            });
        } elseif ($walletId = $this->filters['performance_wallet_id'] ?? null) {
            $query->whereHas('initiativeProject.initiative', function ($q) use ($walletId) {
                $q->where('performance_wallet_id', $walletId);
            });
        } elseif ($departmentId = $this->filters['department_id'] ?? null) {
            $query->whereHas('initiativeProject.initiative.performanceWallet.performanceCard', function ($q) use ($departmentId) {
                $q->where('department_id', $departmentId);
            });
        }

        return $query->orderBy('name')
            ->limit(200)
            ->get()
            ->mapWithKeys(fn ($pc) => [$pc->id => $pc->name])
            ->toArray();
    }
}
