<?php

namespace App\Filament\Widgets;

use App\Models\DonorFinancialPortfolio;
use BezhanSalleh\FilamentShield\Traits\HasWidgetShield;
use Filament\Support\RawJs;
use Filament\Widgets\Concerns\InteractsWithPageFilters;
use Leandrocfe\FilamentApexCharts\Widgets\ApexChartWidget;

class FinancialPortfoliosValueChart extends ApexChartWidget
{
    use HasWidgetShield, InteractsWithPageFilters;

    protected static ?string $chartId = 'financialPortfoliosValueChart';

    protected static ?string $heading = null;

    protected static ?int $sort = 2;

    public function getHeading(): string
    {
        return __('pages.financial_portfolios_dashboard').' - '.__('widgets.financial_portfolios_values');
    }

    protected function getOptions(): array
    {
        // Get all donor financial portfolios
        $portfolios = DonorFinancialPortfolio::orderBy('balance', 'desc')->get();

        $data = [];
        $categories = [];

        // Material Design color palette
        $colors = [
            '#3B82F6', // Blue
            '#10B981', // Emerald
            '#F59E0B', // Amber
            '#EF4444', // Red
            '#8B5CF6', // Violet
            '#EC4899', // Pink
            '#06B6D4', // Cyan
            '#F97316', // Orange
            '#6366F1', // Indigo
            '#84CC16', // Lime
        ];

        foreach ($portfolios as $portfolio) {
            $data[] = (float) $portfolio->balance;
            $categories[] = $portfolio->name;
        }

        // Handle empty data case
        if (empty($data)) {
            return [
                'chart' => [
                    'type' => 'bar',
                    'height' => 400,
                    'fontFamily' => 'inherit',
                    'toolbar' => [
                        'show' => false,
                    ],
                ],
                'series' => [
                    [
                        'name' => __('widgets.portfolio_value'),
                        'data' => [0],
                    ],
                ],
                'xaxis' => [
                    'categories' => [__('widgets.no_portfolios')],
                    'labels' => [
                        'style' => [
                            'fontFamily' => 'inherit',
                            'fontWeight' => 500,
                            'fontSize' => '12px',
                        ],
                    ],
                ],
                'colors' => ['#E5E7EB'],
            ];
        }

        return [
            'chart' => [
                'type' => 'bar',
                'height' => 400,
                'fontFamily' => 'inherit',
                'toolbar' => [
                    'show' => false,
                ],
                'direction' => 'rtl',
            ],
            'series' => [
                [
                    'name' => __('widgets.portfolio_value'),
                    'data' => $data,
                ],
            ],
            'xaxis' => [
                'categories' => $categories,
                'labels' => [
                    'style' => [
                        'fontFamily' => 'inherit',
                        'fontWeight' => 500,
                        'fontSize' => '12px',
                    ],
                    'rotate' => -45,
                    'maxHeight' => 120,
                ],
                'title' => [
                    'text' => __('widgets.portfolio_names'),
                    'style' => [
                        'fontFamily' => 'inherit',
                        'fontWeight' => 600,
                        'fontSize' => '14px',
                    ],
                ],
            ],
            'yaxis' => [
                'title' => [
                    'text' => __('widgets.portfolio_value'),
                    'style' => [
                        'fontFamily' => 'inherit',
                        'fontWeight' => 600,
                        'fontSize' => '14px',
                    ],
                ],
                'labels' => [
                    'style' => [
                        'fontFamily' => 'inherit',
                        'fontSize' => '12px',
                    ],
                ],
            ],
            'colors' => [$colors[0]],
            'dataLabels' => [
                'enabled' => true,
                'style' => [
                    'fontFamily' => 'inherit',
                    'fontWeight' => 600,
                    'fontSize' => '10px',
                    'colors' => ['#ffffff'],
                ],
            ],
            'plotOptions' => [
                'bar' => [
                    'borderRadius' => 4,
                    'columnWidth' => '60%',
                ],
            ],
            'grid' => [
                'show' => true,
                'borderColor' => '#E5E7EB',
                'strokeDashArray' => 4,
                'xaxis' => [
                    'lines' => [
                        'show' => false,
                    ],
                ],
                'yaxis' => [
                    'lines' => [
                        'show' => true,
                    ],
                ],
            ],
            'tooltip' => [
                'enabled' => true,
                'style' => [
                    'fontFamily' => 'inherit',
                ],
            ],
        ];
    }

    protected function extraJsOptions(): ?RawJs
    {
        return RawJs::make(<<<'JS'
        {
            dataLabels: {
                formatter: function (val) {
                    return new Intl.NumberFormat('ar', {
                        minimumFractionDigits: 0,
                        maximumFractionDigits: 0,
                        notation: 'compact',
                        compactDisplay: 'short'
                    }).format(val);
                }
            },
            yaxis: {
                labels: {
                    formatter: function (val) {
                        return new Intl.NumberFormat('ar', {
                            minimumFractionDigits: 0,
                            maximumFractionDigits: 0,
                            notation: 'compact',
                            compactDisplay: 'short'
                        }).format(val);
                    }
                }
            },
            tooltip: {
                y: {
                    formatter: function (val) {
                        return new Intl.NumberFormat('ar', {
                            style: 'decimal',
                            minimumFractionDigits: 0,
                            maximumFractionDigits: 0
                        }).format(val);
                    }
                }
            }
        }
        JS);
    }
}
