<?php

namespace App\Filament\Resources\WellLicenseResource\Pages;

use App\Enums\LicenseStatus;
use App\Filament\Imports\WellLicenseExcelImport;
use App\Filament\Resources\WellLicenseResource;
use App\Filament\Widgets\WellLicensesMap;
use App\Models\WellLicense;
use Archilex\AdvancedTables\AdvancedTables;
use EightyNine\ExcelImport\ExcelImportAction;
use Filament\Actions;
use Filament\Forms\Components\Actions\Action;
use Filament\Notifications\Notification;
use Filament\Resources\Components\Tab;
use Filament\Resources\Pages\ListRecords;
use Illuminate\Database\Eloquent\Builder;

class ListWellLicenses extends ListRecords
{
    use AdvancedTables;

    protected static string $resource = WellLicenseResource::class;

    protected function getHeaderActions(): array
    {
        return [
            ExcelImportAction::make()
                ->label(__('well_license.import'))
                ->color('primary')
                ->use(WellLicenseExcelImport::class)
                ->validateUsing([
                    'association' => 'required|string',
                    'well_name' => 'required|string|max:255',
                    'state' => 'required|string',
                    'city' => 'required|string',
                    'center' => 'required|string',
                    'lat' => 'required|numeric|between:-90,90',
                    'lng' => 'required|numeric|between:-180,180',
                    'well_depth' => 'required|numeric|min:0',
                    'well_diameter' => 'nullable|numeric|min:0',
                    'pump_size' => 'nullable|numeric|min:0',
                    'withdrawal_amount' => 'nullable|numeric|min:0',
                ])
                ->sampleExcel(
                    sampleData: [
                        [
                            'association' => 'جمعية رقم 1',
                            'well_name' => 'اسم البئر 1',
                            'state' => 'القصيم',
                            'city' => 'البدائع',
                            'center' => 'الأبرق',
                            'lat' => 25.981181,
                            'lng' => 43.733409,
                            'well_depth' => 100.50,
                            'well_diameter' => 12.00,
                            'pump_size' => 5.00,
                            'withdrawal_amount' => 1000.00,
                        ],
                        [
                            'association' => 'جمعية رقم 2',
                            'well_name' => 'اسم البئر 2',
                            'state' => 'القصيم',
                            'city' => 'البدائع',
                            'center' => 'الأبرق',
                            'lat' => 24.8000,
                            'lng' => 46.7000,
                            'well_depth' => 150.00,
                            'well_diameter' => 15.00,
                            'pump_size' => 7.50,
                            'withdrawal_amount' => 1500.00,
                        ],
                    ],
                    fileName: 'well_licenses_sample.xlsx',
                    customiseActionUsing: fn (Action $action) => $action
                        ->label(__('well_license.download_sample'))
                        ->icon('heroicon-m-arrow-down-tray'),
                )
                ->modalHeading(__('well_license.import'))
                ->modalDescription(__('well_license.import_description'))
                ->uploadField(
                    fn ($upload) => $upload
                        ->label(__('well_license.import_file'))
                )
                ->beforeImport(function ($data, $livewire) {
                    Notification::make()
                        ->title(__('well_license.import_started'))
                        ->body(__('well_license.import_started_body'))
                        ->info()
                        ->send();
                })
                ->afterImport(function ($data, $livewire) {
                    Notification::make()
                        ->title(__('well_license.import_completed_notification'))
                        ->body(__('well_license.import_completed_notification_body'))
                        ->success()
                        ->send();
                }),
            Actions\CreateAction::make()
                ->label(__('well_license.create_well_license')),
        ];
    }

    protected function getHeaderWidgets(): array
    {
        return [
            WellLicensesMap::class,
        ];
    }

    public function getTabs(): array
    {
        $tabs = [];

        // All tab
        $tabs['all'] = Tab::make(__('well_license.all'))
            ->badge(WellLicense::count());

        // Status-specific tabs
        $statusTabs = [
            'initial_approval' => [
                'status' => LicenseStatus::INITIAL_APPROVAL,
                'label' => __('well_license.initial_approval'),
            ],
            'final_approval' => [
                'status' => LicenseStatus::FINAL_APPROVAL,
                'label' => __('well_license.final_approval'),
            ],
            'approved' => [
                'status' => LicenseStatus::APPROVED,
                'label' => __('well_license.approved'),
            ],
            'rejected' => [
                'status' => LicenseStatus::REJECTED,
                'label' => __('well_license.rejected'),
            ],
        ];

        foreach ($statusTabs as $key => $config) {
            $tabs[$key] = Tab::make($config['label'])
                ->modifyQueryUsing(fn (Builder $query) => $query->where('status', $config['status']->value))
                ->badge(WellLicense::where('status', $config['status']->value)->count());
        }

        return $tabs;
    }
}
