<?php

namespace App\Filament\Resources\ProjectCharterResource\Pages;

use App\Filament\Resources\ProjectCharterResource;
use App\Filament\Resources\ProjectCharterResource\RelationManagers\ChangeRequestsRelationManager;
use App\Filament\Resources\ProjectCharterResource\RelationManagers\EscalationRequestsRelationManager;
use Filament\Actions;
use Filament\Infolists\Components\Grid;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Infolist;
use Filament\Resources\Pages\ViewRecord;

class ManageProjectGovernance extends ViewRecord
{
    protected static string $resource = ProjectCharterResource::class;

    protected static ?string $navigationIcon = 'heroicon-o-clipboard-document-check';

    public static function getNavigationLabel(): string
    {
        return __('project_charter.manage_governance');
    }

    public function getTitle(): string
    {
        return __('project_charter.manage_governance').' - '.$this->getRecord()->name;
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('back_to_charter')
                ->label(__('project_charter.back_to_charter'))
                ->icon('heroicon-o-arrow-left')
                ->color('gray')
                ->url(fn () => ProjectCharterResource::getUrl('view', ['record' => $this->getRecord()])),
        ];
    }

    public function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([
                $this->getGovernanceOverviewSection(),
            ]);
    }

    protected function getGovernanceOverviewSection(): Section
    {
        return Section::make(__('project_charter.governance_overview'))
            ->icon('heroicon-o-clipboard-document-check')
            ->schema([
                Grid::make(3)
                    ->schema([
                        TextEntry::make('status')
                            ->label(__('project_charter.status'))
                            ->badge()
                            ->color(fn ($record) => match ($record->status) {
                                'approved' => 'success',
                                'draft' => 'warning',
                                'awaiting_approval' => 'info',
                                'rejected' => 'danger',
                                'completed' => 'success',
                                'canceled' => 'gray',
                                'suspended' => 'gray',
                                default => 'gray',
                            })
                            ->formatStateUsing(fn ($state) => \App\Models\ProjectCharter::getStatuses()[$state] ?? $state),

                        TextEntry::make('change_requests_count')
                            ->label(__('project_charter.change_requests_count'))
                            ->state(fn ($record) => $record->changeRequests()->count())
                            ->badge()
                            ->color('info'),

                        TextEntry::make('escalation_requests_count')
                            ->label(__('project_charter.escalation_requests_count'))
                            ->state(fn ($record) => $record->escalationRequests()->count())
                            ->badge()
                            ->color('warning'),
                    ]),

                Grid::make(2)
                    ->schema([
                        TextEntry::make('pending_change_requests')
                            ->label(__('project_charter.pending_change_requests'))
                            ->state(fn ($record) => $record->changeRequests()->where('status', 'pending')->count())
                            ->badge()
                            ->color('warning'),

                        TextEntry::make('pending_escalations')
                            ->label(__('project_charter.pending_escalations'))
                            ->state(fn ($record) => $record->escalationRequests()->whereIn('status', ['pending', 'escalated_to_level_2'])->count())
                            ->badge()
                            ->color('danger'),
                    ]),
            ])
            ->columnSpanFull();
    }

    public function getRelationManagers(): array
    {
        return [
            ChangeRequestsRelationManager::class,
            EscalationRequestsRelationManager::class,
        ];
    }
}
