<?php

namespace App\Filament\Resources\ProjectCharterResource\Pages;

use App\Filament\Resources\ProjectCharterResource;
use Filament\Forms;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Form;
use Filament\Resources\Pages\ManageRelatedRecords;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class ManageProjectCharterFiles extends ManageRelatedRecords
{
    protected static string $resource = ProjectCharterResource::class;

    protected static string $relationship = 'files';

    protected static ?string $navigationIcon = 'heroicon-o-document-arrow-up';

    public static function getNavigationLabel(): string
    {
        return __('project_charter_file.manage_files');
    }

    public static function getModelLabel(): string
    {
        return __('project_charter_file.file');
    }

    public static function getPluralModelLabel(): string
    {
        return __('project_charter_file.files');
    }

    public function getTitle(): string
    {
        return __('project_charter_file.files');
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make(__('project_charter_file.file_details'))
                    ->description(__('project_charter_file.file_details_description'))
                    ->schema([
                        Forms\Components\TextInput::make('name')
                            ->label(__('project_charter_file.name'))
                            ->required()
                            ->maxLength(255)
                            ->placeholder(__('project_charter_file.name_placeholder'))
                            ->columnSpanFull(),

                        SpatieMediaLibraryFileUpload::make('charter_file')
                            ->label(__('project_charter_file.file_upload'))
                            ->collection('charter_files')
                            ->visibility('private')
                            ->acceptedFileTypes([
                                'application/pdf',
                                'image/jpeg',
                                'image/png',
                                'image/jpg',
                            ])
                            ->maxSize(10240) // 10MB
                            ->downloadable()
                            ->required()
                            ->helperText(__('project_charter_file.file_helper'))
                            ->columnSpanFull(),
                    ])
                    ->columns(1),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('name')
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label(__('project_charter_file.name'))
                    ->searchable()
                    ->sortable()
                    ->weight('medium'),

                Tables\Columns\TextColumn::make('file_name')
                    ->label(__('project_charter_file.file_name'))
                    ->getStateUsing(fn ($record) => $record->getFirstMedia('charter_files')?->name ?? '-')
                    ->icon('heroicon-o-document')
                    ->placeholder('-'),

                Tables\Columns\TextColumn::make('file_size')
                    ->label(__('project_charter_file.file_size'))
                    ->getStateUsing(fn ($record) => $record->file_size ?? '-')
                    ->placeholder('-'),

                Tables\Columns\IconColumn::make('has_file')
                    ->label(__('project_charter_file.has_file'))
                    ->boolean()
                    ->getStateUsing(fn ($record) => $record->hasFile()),

                Tables\Columns\TextColumn::make('creator.name')
                    ->label(__('project_charter_file.uploaded_by'))
                    ->placeholder('-')
                    ->toggleable(),

                Tables\Columns\TextColumn::make('created_at')
                    ->label(__('project_charter_file.uploaded_at'))
                    ->date()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\Filter::make('has_file')
                    ->label(__('project_charter_file.filter_has_file'))
                    ->query(fn (Builder $query): Builder => $query->has('media')),
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->label(__('project_charter_file.upload_file'))
                    ->icon('heroicon-o-document-arrow-up')
                    ->mutateFormDataUsing(function (array $data): array {
                        $data['project_charter_id'] = $this->getOwnerRecord()->id;

                        return $data;
                    })
                    ->successNotificationTitle(__('project_charter_file.uploaded_successfully')),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\Action::make('download')
                        ->label(__('project_charter_file.download'))
                        ->icon('heroicon-o-arrow-down-tray')
                        ->color('info')
                        ->url(fn ($record) => $record->getFirstMedia('charter_files')?->getUrl())
                        ->openUrlInNewTab()
                        ->visible(fn ($record) => $record->hasFile()),

                    Tables\Actions\ViewAction::make(),
                    Tables\Actions\EditAction::make()
                        ->successNotificationTitle(__('project_charter_file.updated_successfully')),
                    Tables\Actions\DeleteAction::make()
                        ->successNotificationTitle(__('project_charter_file.deleted_successfully')),
                ]),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->successNotificationTitle(__('project_charter_file.deleted_successfully')),
                ]),
            ])
            ->emptyStateIcon('heroicon-o-document-arrow-up')
            ->emptyStateHeading(__('project_charter_file.no_files'))
            ->emptyStateDescription(__('project_charter_file.no_files_description'))
            ->emptyStateActions([
                Tables\Actions\CreateAction::make()
                    ->label(__('project_charter_file.upload_file'))
                    ->icon('heroicon-o-document-arrow-up')
                    ->mutateFormDataUsing(function (array $data): array {
                        $data['project_charter_id'] = $this->getOwnerRecord()->id;

                        return $data;
                    })
                    ->successNotificationTitle(__('project_charter_file.uploaded_successfully')),
            ])
            ->defaultSort('created_at', 'desc');
    }
}
