<?php

namespace App\Filament\Resources\ProjectCharterChangeRequestResource\Pages;

use App\Filament\Resources\ProjectCharterChangeRequestResource;
use App\Models\ProjectCharterChangeRequest;
use Filament\Actions;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\ViewRecord;
use Illuminate\Support\Facades\Auth;

class ViewProjectCharterChangeRequest extends ViewRecord
{
    protected static string $resource = ProjectCharterChangeRequestResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('first_approve')
                ->label(__('project_charter_change_request.first_approve'))
                ->icon('heroicon-o-check-circle')
                ->color('info')
                ->visible(fn (ProjectCharterChangeRequest $record): bool => $record->canFirstApprove())
                ->form([
                    \Filament\Forms\Components\Textarea::make('approval_notes')
                        ->label(__('project_charter_change_request.first_approval_notes'))
                        ->placeholder(__('project_charter_change_request.first_approval_notes_placeholder'))
                        ->maxLength(65535)
                        ->rows(3),
                ])
                ->action(function (ProjectCharterChangeRequest $record, array $data): void {
                    $result = $record->firstApprove($data['approval_notes'] ?? '');

                    if ($result) {
                        Notification::make()
                            ->title(__('project_charter_change_request.change_request_first_approved'))
                            ->success()
                            ->send();

                        // Redirect to the updated record
                        $this->redirect($this->getResource()::getUrl('view', ['record' => $record]));
                    } else {
                        Notification::make()
                            ->title(__('project_charter_change_request.first_approval_failed'))
                            ->danger()
                            ->send();
                    }
                }),

            Actions\Action::make('final_approve')
                ->label(__('project_charter_change_request.final_approve'))
                ->icon('heroicon-o-check-circle')
                ->color('success')
                ->visible(fn (ProjectCharterChangeRequest $record): bool => $record->canFinalApprove())
                ->form([
                    \Filament\Forms\Components\Textarea::make('approval_notes')
                        ->label(__('project_charter_change_request.final_approval_notes'))
                        ->placeholder(__('project_charter_change_request.final_approval_notes_placeholder'))
                        ->maxLength(65535)
                        ->rows(3),

                    SpatieMediaLibraryFileUpload::make('final_approval_attachments')
                        ->label(__('project_charter_change_request.final_approval_attachments'))
                        ->collection('final_approval_attachments')
                        ->visibility('private')
                        ->multiple()
                        ->acceptedFileTypes(['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', 'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', 'image/jpeg', 'image/png', 'image/gif'])
                        ->maxFiles(5)
                        ->helperText(__('project_charter_change_request.final_approval_attachments_help')),
                ])
                ->action(function (ProjectCharterChangeRequest $record, array $data): void {
                    // The SpatieMediaLibraryFileUpload component handles file uploads automatically
                    // when the form is submitted, so we don't need to manually handle attachments here
                    $result = $record->finalApprove($data['approval_notes'] ?? '');

                    if ($result) {
                        Notification::make()
                            ->title(__('project_charter_change_request.change_request_final_approved'))
                            ->success()
                            ->send();

                        // Redirect to the updated record
                        $this->redirect($this->getResource()::getUrl('view', ['record' => $record]));
                    } else {
                        Notification::make()
                            ->title(__('project_charter_change_request.final_approval_failed'))
                            ->danger()
                            ->send();
                    }
                }),

            Actions\Action::make('reject')
                ->label(__('project_charter_change_request.reject'))
                ->icon('heroicon-o-x-circle')
                ->color('danger')
                ->visible(fn (ProjectCharterChangeRequest $record): bool => $record->canReject())
                ->form([
                    \Filament\Forms\Components\Textarea::make('review_notes')
                        ->label(__('project_charter_change_request.review_notes'))
                        ->placeholder(__('project_charter_change_request.rejection_reason_placeholder'))
                        ->required()
                        ->maxLength(65535)
                        ->rows(3),
                ])
                ->action(function (ProjectCharterChangeRequest $record, array $data): void {
                    $result = $record->reject($data['review_notes']);

                    if ($result) {
                        Notification::make()
                            ->title(__('project_charter_change_request.change_request_rejected'))
                            ->success()
                            ->send();

                        // Redirect to the updated record
                        $this->redirect($this->getResource()::getUrl('view', ['record' => $record]));
                    } else {
                        Notification::make()
                            ->title(__('project_charter_change_request.rejection_failed'))
                            ->danger()
                            ->send();
                    }
                }),

            Actions\EditAction::make()
                ->visible(
                    fn (ProjectCharterChangeRequest $record): bool => Auth::user()->can('update', $record) &&
                        $record->status === ProjectCharterChangeRequest::STATUS_PENDING
                ),
        ];
    }

    protected function getHeaderWidgets(): array
    {
        return [
            // ProjectCharterChangeRequestResource\Widgets\ChangeRequestSummaryWidget::class,
        ];
    }
}
