<?php

namespace App\Filament\Resources\PerformanceCardResource\Pages;

use App\Filament\Resources\PerformanceCardResource;
use App\Models\PerformanceCard;
use App\Models\PerformanceWallet;
use Filament\Actions;
use Filament\Forms;
use Filament\Infolists\Components\Grid;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Infolist;
use Filament\Resources\Pages\ViewRecord;
use Filament\Support\Enums\FontWeight;

class ViewPerformanceCard extends ViewRecord
{
    protected static string $resource = PerformanceCardResource::class;

    public static function getNavigationLabel(): string
    {
        return __('performance_card.performance_card');
    }

    protected function getHeaderActions(): array
    {
        return [
            $this->getWalletAction(),
            Actions\EditAction::make(),
            Actions\DeleteAction::make(),
        ];
    }

    protected function getWalletAction(): Actions\Action
    {
        $hasWallet = $this->record->wallet !== null;

        return Actions\Action::make('manage_wallet')
            ->label($hasWallet ? __('performance_wallet.edit_wallet') : __('performance_wallet.create_wallet'))
            ->icon($hasWallet ? 'heroicon-o-pencil-square' : 'heroicon-o-plus-circle')
            ->color($hasWallet ? 'warning' : 'success')
            ->form($this->getWalletForm())
            ->fillForm(function (): array {
                if ($this->record->wallet) {
                    return [
                        'name' => $this->record->wallet->name,
                        'start_date' => $this->record->wallet->start_date,
                        'expected_end_date' => $this->record->wallet->expected_end_date,
                        'description' => $this->record->wallet->description,
                    ];
                }

                return [];
            })
            ->action(function (array $data): void {
                if ($this->record->wallet) {
                    // Update existing wallet
                    $this->record->wallet->update($data);
                    \Filament\Notifications\Notification::make()
                        ->title(__('performance_wallet.updated_successfully'))
                        ->success()
                        ->send();
                } else {
                    // Create new wallet
                    $data['performance_card_id'] = $this->record->id;
                    PerformanceWallet::create($data);
                    \Filament\Notifications\Notification::make()
                        ->title(__('performance_wallet.created_successfully'))
                        ->success()
                        ->send();
                }

                // Refresh the record to show updated wallet information
                $this->record->refresh();
            })
            ->modalHeading($hasWallet ? __('performance_wallet.edit_wallet') : __('performance_wallet.create_wallet'))
            ->modalSubmitActionLabel($hasWallet ? __('filament-actions::edit.single.modal.actions.save.label') : __('filament-actions::create.single.modal.actions.create.label'));
    }

    protected function getWalletForm(): array
    {
        return [
            Forms\Components\TextInput::make('name')
                ->label(__('performance_wallet.name'))
                ->required()
                ->maxLength(255)
                ->helperText(__('performance_wallet.name_helper')),

            Forms\Components\DatePicker::make('start_date')
                ->label(__('performance_wallet.start_date'))
                ->required()
                ->helperText(__('performance_wallet.date_range_helper')),

            Forms\Components\DatePicker::make('expected_end_date')
                ->label(__('performance_wallet.expected_end_date'))
                ->required()
                ->after('start_date'),

            Forms\Components\Textarea::make('description')
                ->label(__('performance_wallet.description'))
                ->required()
                ->rows(4)
                ->maxLength(1000)
                ->helperText(__('performance_wallet.description_helper')),
        ];
    }

    public function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([
                Grid::make(1)
                    ->schema([
                        $this->getPerformanceCardInfoSection(),
                    ]),
            ]);
    }

    public function getPerformanceCardInfoSection(): Section
    {
        return Section::make(__('performance_card.performance_card_information'))
            ->icon('heroicon-o-chart-bar')
            ->schema([
                Grid::make(2)
                    ->schema([
                        TextEntry::make('id')
                            ->label(__('performance_card.id'))
                            ->weight(FontWeight::SemiBold)
                            ->color('gray')
                            ->badge()
                            ->prefix('#'),

                        TextEntry::make('project_manner')
                            ->label(__('performance_card.project_manner'))
                            ->formatStateUsing(fn (string $state): string => PerformanceCard::getProjectManners()[$state] ?? $state)
                            ->badge()
                            ->color(fn (string $state): string => match ($state) {
                                PerformanceCard::PROJECT_MANNER_DEVELOPMENTAL => 'success',
                                PerformanceCard::PROJECT_MANNER_SUPPORT => 'info',
                                PerformanceCard::PROJECT_MANNER_EMPOWERMENT => 'warning',
                                PerformanceCard::PROJECT_MANNER_INVESTMENT => 'primary',
                                default => 'gray',
                            }),

                        TextEntry::make('department.name')
                            ->label(__('performance_card.department'))
                            ->icon('heroicon-o-building-office'),
                    ]),

                TextEntry::make('goal')
                    ->label(__('performance_card.goal'))
                    ->html()
                    ->columnSpanFull(),

                TextEntry::make('products.name')
                    ->label(__('performance_card.products'))
                    ->badge()
                    ->color('gray')
                    ->separator(' ')
                    ->columnSpanFull(),

                TextEntry::make('assignees.name')
                    ->label(__('performance_card.assignees'))
                    ->badge()
                    ->color('primary')
                    ->separator(' ')
                    ->columnSpanFull(),

                TextEntry::make('results.result')
                    ->label(__('performance_card.results'))
                    ->listWithLineBreaks()
                    ->bulleted()
                    ->columnSpanFull()
                    ->placeholder(__('performance_card.no_results')),
            ]);
    }
}
