<?php

namespace App\Filament\Resources;

use App\Filament\Resources\DepartmentResource\Pages;
use App\Models\Department;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class DepartmentResource extends Resource
{
    protected static ?string $model = Department::class;

    protected static ?string $navigationIcon = 'heroicon-o-building-office';

    protected static ?int $navigationSort = 9;

    public static function getNavigationLabel(): string
    {
        return __('department.departments');
    }

    public static function getModelLabel(): string
    {
        return __('department.department');
    }

    public static function getPluralModelLabel(): string
    {
        return __('department.departments');
    }

    public static function getNavigationGroup(): ?string
    {
        return __('common.navigation.group.user_management');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('name')
                    ->label(__('department.name'))
                    ->required()
                    ->maxLength(255)
                    ->unique(ignoreRecord: true)
                    ->helperText(__('department.name_helper')),

                Forms\Components\Textarea::make('description')
                    ->label(__('department.description'))
                    ->rows(3)
                    ->maxLength(1000)
                    ->helperText(__('department.description_helper')),

                Forms\Components\Select::make('users')
                    ->label(__('department.users'))
                    ->relationship('users', 'name', fn (\Illuminate\Database\Eloquent\Builder $query) => $query->whereNull('association_id'))
                    ->multiple()
                    ->preload()
                    ->searchable()
                    ->helperText(__('department.users_helper')),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label(__('department.name'))
                    ->sortable()
                    ->searchable()
                    ->limit(20)
                    ->tooltip(fn ($record) => $record->name)
                    ->toggleable(),

                Tables\Columns\TextColumn::make('description')
                    ->label(__('department.description'))
                    ->limit(20)
                    ->tooltip(fn ($record) => $record->description)
                    ->searchable()
                    ->toggleable(),

                Tables\Columns\TextColumn::make('users_count')
                    ->label(__('department.users_count'))
                    ->counts('users')
                    ->sortable()
                    ->toggleable(),

                Tables\Columns\TextColumn::make('created_at')
                    ->date()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\Filter::make('has_users')
                    ->label(__('department.has_users'))
                    ->query(fn (Builder $query) => $query->has('users'))
                    ->toggle(),
            ])
            ->headerActions([
                Tables\Actions\ExportAction::make()
                    ->exporter(\App\Filament\Exports\DepartmentExporter::class),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\ViewAction::make(),
                    Tables\Actions\EditAction::make(),
                    Tables\Actions\DeleteAction::make(),
                ]),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListDepartments::route('/'),
            'create' => Pages\CreateDepartment::route('/create'),
            'edit' => Pages\EditDepartment::route('/{record}/edit'),
        ];
    }
}
