<?php

namespace App\Filament\Resources\AssociationResource\Pages;

use App\Filament\Resources\AssociationResource;
use App\Models\Association;
use Filament\Actions;
use Filament\Infolists\Components\Grid;
use Filament\Infolists\Components\Group;
use Filament\Infolists\Components\ImageEntry;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\Split;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Infolist;
use Filament\Resources\Pages\ViewRecord;
use Filament\Support\Enums\FontWeight;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\HtmlString;

class ViewAssociation extends ViewRecord
{
    protected static string $resource = AssociationResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\EditAction::make(),
            Actions\DeleteAction::make(),

            Actions\Action::make('verify')
                ->label(__('association.action_verify'))
                ->icon('heroicon-o-shield-check')
                ->color('success')
                ->visible(fn (Association $record): bool => ! $record->is_verified && Auth::user()->can('verify_association'))
                ->requiresConfirmation()
                ->modalHeading(__('association.confirm_verify_title'))
                ->modalDescription(__('association.confirm_verify_description'))
                ->action(fn (Association $record) => $record->update(['is_verified' => true])),

            Actions\Action::make('unverify')
                ->label(__('association.action_unverify'))
                ->icon('heroicon-o-shield-exclamation')
                ->color('warning')
                ->visible(fn (Association $record): bool => $record->is_verified && Auth::user()->can('unverify_association'))
                ->requiresConfirmation()
                ->modalHeading(__('association.confirm_unverify_title'))
                ->modalDescription(__('association.confirm_unverify_description'))
                ->action(fn (Association $record) => $record->update(['is_verified' => false])),

            Actions\Action::make('lock')
                ->label(__('association.action_lock'))
                ->icon('heroicon-o-lock-closed')
                ->color('danger')
                ->visible(fn (Association $record): bool => ! $record->is_locked && Auth::user()->can('lock_profile_association'))
                ->requiresConfirmation()
                ->modalHeading(__('association.confirm_lock_title'))
                ->modalDescription(__('association.confirm_lock_description'))
                ->action(fn (Association $record) => $record->update(['is_locked' => true])),

            Actions\Action::make('unlock')
                ->label(__('association.action_unlock'))
                ->icon('heroicon-o-lock-open')
                ->color('success')
                ->visible(fn (Association $record): bool => $record->is_locked && Auth::user()->can('unlock_profile_association'))
                ->requiresConfirmation()
                ->modalHeading(__('association.confirm_unlock_title'))
                ->modalDescription(__('association.confirm_unlock_description'))
                ->action(fn (Association $record) => $record->update(['is_locked' => false])),
        ];
    }

    public function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([
                Split::make([
                    // Left Column - Main Information
                    Grid::make(1)
                        ->schema([
                            // Association Information Card
                            Section::make(__('association.association_information'))
                                ->icon('heroicon-o-building-library')
                                ->schema([
                                    Grid::make(3)
                                        ->schema([
                                            ImageEntry::make('logo_url')
                                                ->label(__('association.logo'))
                                                ->circular()
                                                ->size(80)
                                                ->defaultImageUrl('/images/default-logo.png'),

                                            Group::make([
                                                TextEntry::make('name')
                                                    ->label(__('association.name'))
                                                    ->weight(FontWeight::Bold)
                                                    ->size(TextEntry\TextEntrySize::Large)
                                                    ->color('primary'),

                                                TextEntry::make('representative_full_name')
                                                    ->label(__('association.representative'))
                                                    ->icon('heroicon-o-user')
                                                    ->color('gray'),

                                                TextEntry::make('representative_position')
                                                    ->label(__('association.position'))
                                                    ->icon('heroicon-o-briefcase')
                                                    ->color('gray'),
                                            ])
                                                ->columnSpan(2),
                                        ]),

                                    Grid::make(2)
                                        ->schema([
                                            TextEntry::make('type')
                                                ->label(__('association.type_label'))
                                                ->badge()
                                                ->icon('heroicon-o-building-office-2'),

                                            TextEntry::make('license_number')
                                                ->label(__('association.license_number'))
                                                ->icon('heroicon-o-document-text')
                                                ->copyable(),

                                            TextEntry::make('license_expiration_date')
                                                ->label(__('association.license_expiration_date'))
                                                ->icon('heroicon-o-calendar')
                                                ->date()
                                                ->color(fn ($state) => $state && $state->isPast() ? 'danger' : 'success'),

                                            TextEntry::make('is_verified')
                                                ->label(__('association.verification_status'))
                                                ->badge()
                                                ->formatStateUsing(fn (bool $state): string => $state ? __('association.verified') : __('association.not_verified'))
                                                ->color(fn (bool $state): string => $state ? 'success' : 'warning')
                                                ->icon(fn (bool $state): string => $state ? 'heroicon-o-shield-check' : 'heroicon-o-shield-exclamation'),

                                            TextEntry::make('is_locked')
                                                ->label(__('association.lock_status'))
                                                ->badge()
                                                ->formatStateUsing(fn (bool $state): string => $state ? __('association.locked') : __('association.unlocked'))
                                                ->color(fn (bool $state): string => $state ? 'danger' : 'success')
                                                ->icon(fn (bool $state): string => $state ? 'heroicon-o-lock-closed' : 'heroicon-o-lock-open'),
                                        ]),
                                ]),

                            // Contact Information Card
                            Section::make(__('association.contact_information'))
                                ->icon('heroicon-o-phone')
                                ->schema([
                                    Grid::make(2)
                                        ->schema([
                                            TextEntry::make('phone_number')
                                                ->label(__('association.phone_number'))
                                                ->icon('heroicon-o-phone')
                                                ->copyable(),

                                            TextEntry::make('email')
                                                ->label(__('association.email'))
                                                ->icon('heroicon-o-envelope')
                                                ->copyable()
                                                ->columnSpanFull(),
                                        ]),
                                ]),

                            // National Address Card
                            Section::make(__('association.national_address'))
                                ->icon('heroicon-o-map-pin')
                                ->schema([

                                    Grid::make(3)
                                        ->schema([
                                            TextEntry::make('city')
                                                ->label(__('association.city'))
                                                ->placeholder('—'),

                                            TextEntry::make('postal_code')
                                                ->label(__('association.postal_code'))
                                                ->placeholder('—'),

                                            TextEntry::make('neighborhood')
                                                ->label(__('association.neighborhood'))
                                                ->placeholder('—'),

                                            TextEntry::make('building_number')
                                                ->label(__('association.building_number'))
                                                ->placeholder('—'),

                                            TextEntry::make('street')
                                                ->label(__('association.street'))
                                                ->placeholder('—'),

                                            TextEntry::make('branch_number')
                                                ->label(__('association.branch_number'))
                                                ->placeholder('—'),
                                        ]),
                                ]),
                        ]),

                    // Right Column - Additional Information
                    Grid::make(1)
                        ->schema([
                            // Geographic Scope Card
                            Section::make(__('association.geographic_scope'))
                                ->icon('heroicon-o-globe-europe-africa')
                                ->schema([
                                    TextEntry::make('states.name')
                                        ->label(__('association.states'))
                                        ->badge()
                                        ->separator(',')
                                        ->color('gray')
                                        ->placeholder('—'),

                                    TextEntry::make('cities.name')
                                        ->label(__('association.cities'))
                                        ->badge()
                                        ->separator(',')
                                        ->color('gray')
                                        ->placeholder('—'),

                                    TextEntry::make('centers.name')
                                        ->label(__('association.centers'))
                                        ->badge()
                                        ->separator(',')
                                        ->color('gray')
                                        ->placeholder('—'),

                                    TextEntry::make('residentialGatherings.name')
                                        ->label(__('order.residential_gatherings'))
                                        ->badge()
                                        ->color('info')
                                        ->separator(' '),

                                    // TextEntry::make('location')
                                    //     ->label(__('association.location'))
                                    //     ->formatStateUsing(function ($state) {
                                    //         if (!$state || !is_array($state) || !isset($state['lat'], $state['lng'])) {
                                    //             return '—';
                                    //         }
                                    //         return "Lat: {$state['lat']}, Lng: {$state['lng']}";
                                    //     })
                                    //     ->icon('heroicon-o-map-pin')
                                    //     ->placeholder('—'),

                                    TextEntry::make('location')
                                        ->label(__('association.location'))
                                        ->icon('heroicon-o-globe-alt')
                                        ->columnSpanFull()
                                        ->color('primary'),
                                ]),

                            // Banking Details Card
                            Section::make(__('association.banking_details'))
                                ->icon('heroicon-o-banknotes')
                                ->schema([
                                    TextEntry::make('iban')
                                        ->label(__('association.iban'))
                                        ->icon('heroicon-o-credit-card')
                                        ->copyable()
                                        ->placeholder('—'),

                                    TextEntry::make('bank_name')
                                        ->label(__('association.bank_name'))
                                        ->icon('heroicon-o-building-office')
                                        ->placeholder('—'),

                                    TextEntry::make('account_holder_name')
                                        ->label(__('association.account_holder_name'))
                                        ->icon('heroicon-o-user')
                                        ->placeholder('—'),
                                ]),

                            // Documents Card
                            Section::make(__('association.documents'))
                                ->icon('heroicon-o-document-text')
                                ->schema([
                                    Grid::make(1)
                                        ->schema([
                                            TextEntry::make('authorization_letter_url')
                                                ->label(__('association.authorization_letter'))
                                                ->formatStateUsing(fn ($state) => $state ? new HtmlString('<a class="text-blue-500 hover:underline" href="'.$state.'" target="_blank">'.
                                                    __('common.download').'</a>') : __('association.document_not_available'))
                                                ->icon('heroicon-o-document-text')
                                                ->color(fn ($state) => $state ? 'success' : 'gray'),

                                            TextEntry::make('license_certificate_url')
                                                ->label(__('association.license_certificate'))
                                                ->formatStateUsing(fn ($state) => $state ? new HtmlString('<a class="text-blue-500 hover:underline" href="'.$state.'" target="_blank">'.
                                                    __('common.download').'</a>') : __('association.document_not_available'))
                                                ->icon('heroicon-o-document-text')
                                                ->color(fn ($state) => $state ? 'success' : 'gray'),

                                            TextEntry::make('iban_certificate_url')
                                                ->label(__('association.iban_certificate'))
                                                ->formatStateUsing(fn ($state) => $state ? new HtmlString('<a class="text-blue-500 hover:underline" href="'.$state.'" target="_blank">'.
                                                    __('common.download').'</a>') : __('association.document_not_available'))
                                                ->icon('heroicon-o-document-text')
                                                ->color(fn ($state) => $state ? 'success' : 'gray'),

                                            TextEntry::make('national_address_certificate_url')
                                                ->label(__('association.national_address_certificate'))
                                                ->formatStateUsing(fn ($state) => $state ? new HtmlString('<a class="text-blue-500 hover:underline" href="'.$state.'" target="_blank">'.
                                                    __('common.download').'</a>') : __('association.document_not_available'))
                                                ->icon('heroicon-o-document-text')
                                                ->color(fn ($state) => $state ? 'success' : 'gray'),

                                            TextEntry::make('governance_certificate_url')
                                                ->label(__('association.governance_certificate'))
                                                ->formatStateUsing(fn ($state) => $state ? new HtmlString('<a class="text-blue-500 hover:underline" href="'.$state.'" target="_blank">'.
                                                    __('common.download').'</a>') : __('association.document_not_available'))
                                                ->icon('heroicon-o-document-text')
                                                ->color(fn ($state) => $state ? 'success' : 'gray'),

                                            TextEntry::make('financial_reports_url')
                                                ->label(__('association.financial_reports'))
                                                ->formatStateUsing(fn ($state) => $state ? new HtmlString('<a class="text-blue-500 hover:underline" href="'.$state.'" target="_blank">'.
                                                    __('common.download').'</a>') : __('association.document_not_available'))
                                                ->icon('heroicon-o-document-text')
                                                ->color(fn ($state) => $state ? 'success' : 'gray'),
                                        ]),
                                ]),

                            // Metadata Card
                            Section::make(__('common.metadata'))
                                ->icon('heroicon-o-information-circle')
                                ->schema([
                                    Grid::make(2)
                                        ->schema([
                                            TextEntry::make('created_at')
                                                ->label(__('common.created_at'))
                                                ->date()
                                                ->icon('heroicon-o-calendar'),

                                            TextEntry::make('updated_at')
                                                ->label(__('common.updated_at'))
                                                ->date()
                                                ->icon('heroicon-o-calendar'),
                                        ]),
                                ]),
                        ]),
                ])->from('md')
                    ->columnSpanFull(),
            ]);
    }
}
