<?php

namespace App\Filament\Pages;

use App\Models\Department;
use App\Models\Initiative;
use App\Models\PerformanceWallet;
use BezhanSalleh\FilamentShield\Traits\HasPageShield;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Form;
use Filament\Pages\Dashboard as BaseDashboard;
use Filament\Pages\Dashboard\Concerns\HasFiltersForm;

class ProjectsDashboard extends BaseDashboard
{
    use HasFiltersForm, HasPageShield;

    protected static string $routePath = 'projects_dashboard';

    public static function getNavigationLabel(): string
    {
        return __('pages.projects_dashboard');
    }

    public function getTitle(): string
    {
        return __('pages.projects_dashboard');
    }

    protected function getHeaderWidgets(): array
    {
        return [
            // \App\Filament\Widgets\PerformanceDashboardStatsOverview::class,
        ];
    }

    public static function getNavigationGroup(): ?string
    {
        return __('common.navigation.group.project_management');
    }

    public function getWidgets(): array
    {
        return [
            \App\Filament\Widgets\PerformanceDashboardStatsOverview::class,
            \App\Filament\Widgets\ProjectCostOverview::class,
            \App\Filament\Widgets\ProjectCharterStatusChart::class,
            \App\Filament\Widgets\ProjectCostChart::class,
            \App\Filament\Widgets\ProjectProgressGauge::class,
            \App\Filament\Widgets\ProjectRiskMatrixChart::class,
            \App\Filament\Widgets\PerformanceWalletStatusTable::class,
            \App\Filament\Widgets\InitiativeOperationGoalsChart::class,
            \App\Filament\Widgets\ProjectStatusChart::class,
            \App\Filament\Widgets\InitiativeOperationGoalsStatusTable::class,
            \App\Filament\Widgets\InitiativeResultsStatusTable::class,
            \App\Filament\Widgets\ProjectDetailsTable::class,
            \App\Filament\Widgets\ProjectFinancialStatusTable::class,

        ];
    }

    public function filtersForm(Form $form): Form
    {
        // Generate year options from 2020 to current year + 1
        $currentYear = now()->year;
        $years = [];
        for ($year = 2020; $year <= $currentYear + 1; $year++) {
            $years[$year] = $year;
        }

        return $form
            ->schema([
                Section::make(__('performance_dashboard.filters'))
                    ->schema([
                        Select::make('year')
                            ->label(__('widgets.year'))
                            ->placeholder(__('widgets.all_years'))
                            ->options($years)
                            ->searchable()
                            ->native(false),

                        Select::make('department_id')
                            ->label(__('performance_dashboard.department'))
                            ->placeholder(__('performance_dashboard.all_departments'))
                            ->options(
                                Department::all()
                                    ->mapWithKeys(function ($department) {
                                        return [$department->id => $department->name];
                                    })
                                    ->toArray()
                            )
                            ->searchable()
                            ->native(false)
                            ->reactive()
                            ->afterStateUpdated(function (callable $set) {
                                // Clear wallet and initiative filters when department changes
                                $set('performance_wallet_id', null);
                                $set('initiative_id', null);
                            }),

                        Select::make('performance_wallet_id')
                            ->label(__('performance_dashboard.performance_wallet'))
                            ->placeholder(__('performance_dashboard.all_wallets'))
                            ->options(function (callable $get) {
                                $departmentId = $get('department_id');

                                $query = PerformanceWallet::with('performanceCard');
                                if ($departmentId) {
                                    $query->whereHas('performanceCard', function ($q) use ($departmentId) {
                                        $q->where('department_id', $departmentId);
                                    });
                                }

                                return $query->get()
                                    ->mapWithKeys(function ($wallet) {
                                        $name = $wallet->name ?? "Wallet #{$wallet->id}";

                                        return [$wallet->id => $name];
                                    })
                                    ->toArray();
                            })
                            ->searchable()
                            ->native(false)
                            ->reactive()
                            ->afterStateUpdated(function (callable $set) {
                                // Clear initiative filter when wallet changes
                                $set('initiative_id', null);
                            }),

                        Select::make('initiative_id')
                            ->label(__('performance_dashboard.initiative'))
                            ->placeholder(__('performance_dashboard.all_initiatives'))
                            ->options(function (callable $get) {
                                $walletId = $get('performance_wallet_id');

                                $query = Initiative::query();
                                if ($walletId) {
                                    $query->where('performance_wallet_id', $walletId);
                                }

                                return $query->get()
                                    ->mapWithKeys(function ($initiative) {
                                        return [$initiative->id => $initiative->name];
                                    })
                                    ->toArray();
                            })
                            ->searchable()
                            ->native(false)
                            ->reactive(),
                    ])
                    ->columns(4),
            ]);
    }
}
