<?php

namespace App\Filament\Exports;

use App\Models\ExchangeRequest;
use Filament\Actions\Exports\ExportColumn;
use Filament\Actions\Exports\Exporter;
use Filament\Actions\Exports\Models\Export;

class ExchangeRequestExporter extends Exporter
{
    protected static ?string $model = ExchangeRequest::class;

    public static function getColumns(): array
    {
        return [
            ExportColumn::make('id')
                ->label(__('exchange_request.number')),
            ExportColumn::make('status')
                ->label(__('exchange_request.status'))
                ->state(function (ExchangeRequest $record): string {
                    return ExchangeRequest::getStatuses()[$record->status] ?? $record->status;
                }),
            ExportColumn::make('completionReport.description')
                ->label(__('completion_report.description'))
                ->state(function (ExchangeRequest $record): ?string {
                    return $record->completionReport ? $record->completionReport->description : null;
                }),
            ExportColumn::make('completionReport.value')
                ->label(__('completion_report.value'))
                ->state(function (ExchangeRequest $record): ?string {
                    return $record->completionReport ? number_format((float) $record->completionReport->value, 2) : null;
                }),
            ExportColumn::make('transaction_date')
                ->label(__('exchange_request.transaction_date')),
            ExportColumn::make('completionReport.projectCharterDeliverable.name')
                ->label(__('project_charter_deliverable.name'))
                ->state(function (ExchangeRequest $record): ?string {
                    return $record->completionReport && $record->completionReport->projectCharterDeliverable
                        ? $record->completionReport->projectCharterDeliverable->name
                        : null;
                }),
            // completion report approved at
            ExportColumn::make('completionReport.approved_at')
                ->label(__('exchange_request.completion_report_approved_at'))
                ->state(function (ExchangeRequest $record): ?string {
                    return $record->completionReport ? $record->completionReport->approved_at : null;
                }),
            ExportColumn::make('order.id')
                ->label(__('order.id'))
                ->state(function (ExchangeRequest $record): ?string {
                    return $record->order ? '#'.$record->order->id : null;
                }),
            ExportColumn::make('order.type')
                ->label(__('order.order_type'))
                ->state(function (ExchangeRequest $record): ?string {
                    if (! $record->order) {
                        return null;
                    }
                    $orderTypes = \App\Models\Order::getAllOrderTypes();

                    return isset($orderTypes[$record->order->type]) ? __($orderTypes[$record->order->type]) : $record->order->type;
                }),
            ExportColumn::make('association.name')
                ->label(__('order.association'))
                ->state(function (ExchangeRequest $record): ?string {
                    return $record->association ? $record->association->name : null;
                }),
            // charter id and name
            ExportColumn::make('projectCharter.id')
                ->label(__('exchange_request.project_charter_id'))
                ->state(function (ExchangeRequest $record): ?string {
                    if (! $record->projectCharter) {
                        return null;
                    }

                    $projectManner = $record->projectCharter->project_manner;
                    if ($projectManner) {
                        return strtoupper(substr($projectManner, 0, 1)).'-'.$record->projectCharter->id;
                    }

                    return '#'.$record->projectCharter->id;
                }),
            ExportColumn::make('projectCharter.name')
                ->label(__('exchange_request.project_charter_name'))
                ->state(function (ExchangeRequest $record): ?string {
                    return $record->projectCharter ? $record->projectCharter->name : null;
                }),
            ExportColumn::make('createdBy.name')
                ->label(__('exchange_request.created_by')),
            ExportColumn::make('created_at')
                ->label(__('exchange_request.created_at')),
            ExportColumn::make('updated_at')
                ->label(__('exchange_request.updated_at')),
        ];
    }

    public static function getCompletedNotificationBody(Export $export): string
    {
        $successfulRows = $export->successful_rows;
        $rowWord = $successfulRows === 1 ? __('exchange_request.row') : __('exchange_request.rows');

        $body = __('exchange_request.export_completed', [
            'count' => number_format($successfulRows),
            'rows' => $rowWord,
        ]);

        if ($failedRowsCount = $export->getFailedRowsCount()) {
            $failedRowWord = $failedRowsCount === 1 ? __('exchange_request.row') : __('exchange_request.rows');
            $body .= __('exchange_request.export_failed_rows', [
                'count' => number_format($failedRowsCount),
                'rows' => $failedRowWord,
            ]);
        }

        return $body;
    }
}
