<?php

namespace App\Filament\AssociationAdmin\Resources\AssociationProjectCharterResource\Pages;

use App\Filament\AssociationAdmin\Resources\AssociationProjectCharterResource;
use App\Models\Order;
use App\Models\ProjectCharter;
use Filament\Infolists\Components\Grid;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\Split;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Infolist;
use Filament\Resources\Pages\ViewRecord;
use Filament\Support\Enums\FontWeight;
use Illuminate\Support\HtmlString;

class ViewAssociationProjectCharter extends ViewRecord
{
    protected static string $resource = AssociationProjectCharterResource::class;

    public static function getNavigationLabel(): string
    {
        return __('project_charter.charter_details');
    }

    protected function getHeaderActions(): array
    {
        return [];
    }

    public function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([
                $this->getProjectCharterInfoSplit(),
            ]);
    }

    public function getProjectCharterInfoSplit(): Split
    {
        return Split::make([
            // Left Column - Charter Overview
            Grid::make(1)
                ->schema([
                    // Charter Overview Card
                    Section::make(__('project_charter.charter_overview'))
                        ->icon('heroicon-o-document-text')
                        ->schema([
                            Grid::make(2)
                                ->schema([
                                    TextEntry::make('id')
                                        ->label(__('project_charter.charter_id'))
                                        ->weight(FontWeight::SemiBold)
                                        ->color('gray')
                                        ->badge()
                                        ->formatStateUsing(function ($state, $record) {
                                            $projectManner = $record->project_manner;
                                            if ($projectManner) {
                                                return strtoupper(substr($projectManner, 0, 1)).'-'.$state;
                                            }
                                            return '#'.$state;
                                        }),

                                    TextEntry::make('status')
                                        ->label(__('project_charter.status'))
                                        ->badge()
                                        ->formatStateUsing(fn (string $state): string => ProjectCharter::getStatuses()[$state] ?? $state)
                                        ->color(fn (string $state): string => match ($state) {
                                            ProjectCharter::STATUS_APPROVED => 'success',
                                            default => 'gray',
                                        }),

                                    TextEntry::make('name')
                                        ->label(__('project_charter.name'))
                                        ->weight(FontWeight::SemiBold)
                                        ->icon('heroicon-o-document-text'),

                                    TextEntry::make('order.name')
                                        ->label(__('project_charter.linked_order'))
                                        ->formatStateUsing(fn ($record) => $record->order ?
                                            "#{$record->order->id} - ".Order::getOrderTypes()[$record->order->type] : '-')
                                        ->icon('heroicon-o-document-duplicate')
                                        ->color('info')
                                        ->placeholder('-'),
                                ]),

                            TextEntry::make('description')
                                ->label(__('project_charter.description'))
                                ->html()
                                ->columnSpanFull()
                                ->placeholder(__('project_charter.no_description')),

                        ]),

                    // Project Relationships Card
                    Section::make(__('project_charter.project_relationships'))
                        ->icon('heroicon-o-link')
                        ->schema([
                            Grid::make(2)
                                ->schema([
                                    TextEntry::make('initiativeProject.name')
                                        ->label(__('project_charter.initiative_project'))
                                        ->weight(FontWeight::Medium)
                                        ->icon('heroicon-o-rocket-launch')
                                        ->color('primary'),

                                    TextEntry::make('order.id')
                                        ->label(__('project_charter.linked_order'))
                                        ->formatStateUsing(fn ($record) => $record->order ?
                                            "#{$record->order->id} - ".Order::getOrderTypes()[$record->order->type] : '-')
                                        ->icon('heroicon-o-document-duplicate')
                                        ->color('info')
                                        ->placeholder('-'),

                                    TextEntry::make('initiativeProject.initiative.name')
                                        ->label(__('project_charter.initiative'))
                                        ->icon('heroicon-o-light-bulb')
                                        ->color('warning'),

                                    TextEntry::make('initiativeProject.initiative.performanceCard.department.name')
                                        ->label(__('project_charter.department'))
                                        ->icon('heroicon-o-building-office')
                                        ->color('success')
                                        ->placeholder('-'),
                                ]),
                        ]),
                ]),

            // Right Column - Timeline & Financial Information
            Grid::make(1)
                ->schema([
                    // Timeline Card
                    Section::make(__('project_charter.timeline'))
                        ->icon('heroicon-o-calendar-days')
                        ->schema([
                            Grid::make(2)
                                ->schema([
                                    TextEntry::make('start_date')
                                        ->label(__('project_charter.start_date'))
                                        ->date()
                                        ->icon('heroicon-o-calendar-days')
                                        ->color('success')
                                        ->weight(FontWeight::Medium),

                                    TextEntry::make('expected_end_date')
                                        ->label(__('project_charter.expected_end_date'))
                                        ->date()
                                        ->icon('heroicon-o-calendar')
                                        ->color('warning')
                                        ->weight(FontWeight::Medium),
                                ]),

                            // Duration calculation
                            TextEntry::make('duration')
                                ->label(__('project_charter.duration'))
                                ->state(function ($record) {
                                    if (! $record->start_date || ! $record->expected_end_date) {
                                        return '-';
                                    }
                                    $duration = $record->start_date->diffInDays($record->expected_end_date);

                                    return __('project_charter.duration_days', ['days' => $duration]);
                                })
                                ->icon('heroicon-o-clock')
                                ->color('info')
                                ->columnSpanFull(),
                        ]),

                    // Financial Information Card
                    Section::make(__('project_charter.financial_information'))
                        ->icon('heroicon-o-banknotes')
                        ->schema([
                            Grid::make(2)
                                ->schema([
                                    TextEntry::make('estimated_cost')
                                        ->label(__('project_charter.estimated_cost'))
                                        ->numeric(2)
                                        ->prefix(new HtmlString('<span class="icon-saudi_riyal"></span>'))
                                        ->icon('heroicon-o-calculator')
                                        ->color('info')
                                        ->weight(FontWeight::Medium),

                                    TextEntry::make('total_cost')
                                        ->label(__('project_charter.total_cost'))
                                        ->numeric(2)
                                        ->prefix(new HtmlString('<span class="icon-saudi_riyal"></span>'))
                                        ->icon('heroicon-o-banknotes')
                                        ->color('success')
                                        ->weight(FontWeight::Bold),
                                ]),
                        ]),
                ]),
        ])
            ->from('lg')
            ->columnSpan('full');
    }

    protected function getAllRelationManagers(): array
    {
        return [];
    }
}
