<?php

namespace Tests\Feature;

use App\Models\Order;
use App\Models\Org;
use App\Workflows\Models\StepTemplate;
use App\Workflows\Models\WorkflowTemplate;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class WorkflowPermissionStabilityTest extends TestCase
{
    use RefreshDatabase;

    public function test_step_permission_names_are_stable_with_system_key(): void
    {
        // Create a workflow template
        $org = Org::factory()->create();
        $workflowTemplate = WorkflowTemplate::factory()->create([
            'org_id' => $org->id,
            'subject_type' => Order::class,
            'name' => 'Order Processing',
        ]);

        // Create step template with system key
        $stepTemplate = StepTemplate::factory()->create([
            'workflow_template_id' => $workflowTemplate->id,
            'position' => 1,
            'system_key' => 'approval',
            'label' => 'Manager Approval',
        ]);

        $expectedPermissionName = "workflow_{$org->id}_Order_approval";
        $this->assertEquals($expectedPermissionName, $stepTemplate->getPermissionName());

        // Permission should be the same even with different database ID
        $anotherStepTemplate = StepTemplate::factory()->create([
            'workflow_template_id' => $workflowTemplate->id,
            'position' => 1,
            'system_key' => 'approval',
            'label' => 'Manager Approval',
        ]);

        $this->assertEquals($expectedPermissionName, $anotherStepTemplate->getPermissionName());
        $this->assertEquals(
            $stepTemplate->getPermissionName(),
            $anotherStepTemplate->getPermissionName()
        );
    }

    public function test_step_permission_names_are_stable_without_system_key(): void
    {
        // Create a workflow template
        $org = Org::factory()->create();
        $workflowTemplate = WorkflowTemplate::factory()->create([
            'org_id' => $org->id,
            'subject_type' => Order::class,
            'name' => 'Order Processing',
        ]);

        // Create step template without system key
        $stepTemplate = StepTemplate::factory()->create([
            'workflow_template_id' => $workflowTemplate->id,
            'position' => 2,
            'system_key' => null,
            'label' => 'Quality Review',
        ]);

        $expectedPermissionName = "workflow_{$org->id}_Order_pos2_quality_review";
        $this->assertEquals($expectedPermissionName, $stepTemplate->getPermissionName());
    }

    public function test_permission_labels_are_human_readable(): void
    {
        // Create a workflow template
        $workflowTemplate = WorkflowTemplate::factory()->create([
            'name' => 'Order Processing Workflow',
            'subject_type' => Order::class,
        ]);

        // Create step template
        $stepTemplate = StepTemplate::factory()->create([
            'workflow_template_id' => $workflowTemplate->id,
            'label' => 'Manager Approval',
        ]);

        $expectedLabel = 'Order Processing Workflow → Manager Approval';
        $this->assertEquals($expectedLabel, $stepTemplate->getPermissionLabel());
    }

    public function test_label_slug_creation_handles_special_characters(): void
    {
        // Create a workflow template
        $org = Org::factory()->create();
        $workflowTemplate = WorkflowTemplate::factory()->create([
            'org_id' => $org->id,
            'subject_type' => Order::class,
        ]);

        // Create step template with special characters in label
        $stepTemplate = StepTemplate::factory()->create([
            'workflow_template_id' => $workflowTemplate->id,
            'position' => 1,
            'system_key' => null,
            'label' => 'Review & Approval (Final)',
        ]);

        // Should create a clean slug
        $expectedPermissionName = "workflow_{$org->id}_Order_pos1_review_approval_final";
        $this->assertEquals($expectedPermissionName, $stepTemplate->getPermissionName());
    }
}
