<?php

namespace Database\Seeders;

use App\Models\Association;
use App\Models\City;
use App\Models\GrantArea;
use App\Models\Order;
use App\Models\OrderDeliverable;
use App\Models\Org;
use App\Models\State;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Database\Seeder;

class OrderSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get required models (these should exist from other seeders)
        $org = Org::first() ?? Org::create(['name' => 'مؤسسة سدو']);
        $grantAreas = GrantArea::all();
        $associations = Association::all();
        $states = State::all();
        $cities = City::all();
        $users = User::all();

        // If we don't have the required data, create some basic ones
        if ($grantAreas->isEmpty()) {
            $grantAreas = collect([
                GrantArea::create(['name' => 'رعاية الأيتام', 'org_id' => $org->id]),
                GrantArea::create(['name' => 'الرعاية الصحية', 'org_id' => $org->id]),
            ]);
        }

        if ($associations->isEmpty()) {
            $user = $users->first() ?? User::create([
                'name' => 'مستخدم تجريبي',
                'email' => 'test@example.com',
                'phone' => '+966501111111',
                'password' => bcrypt('password'),
            ]);

            $associations = collect([
                Association::create([
                    'name' => 'جمعية تجريبية',
                    'license_number' => 'LIC-TEST-001',
                    'license_expiration_date' => now()->addYears(2),
                    'user_id' => $user->id,
                    'org_id' => $org->id,
                    'city' => 'الرياض',
                    'street' => 'شارع تجريبي',
                    'district' => 'حي تجريبي',
                    'postal_code' => '12345',
                    'building_number' => '123',
                    'lat' => 24.7136,
                    'lng' => 46.6753,
                    'iban' => 'SA0380000000608010167519',
                    'bank_name' => 'بنك تجريبي',
                    'account_holder_name' => 'جمعية تجريبية',
                    'is_verified' => true,
                    'is_locked' => false,
                ]),
            ]);
        }

        if ($states->isEmpty()) {
            $states = collect([
                State::create(['name' => 'منطقة الرياض']),
            ]);
        }

        if ($cities->isEmpty()) {
            $cities = collect([
                City::create(['name' => 'الرياض', 'state_id' => $states->first()->id]),
            ]);
        }

        $orders = [
            [
                'name' => 'مشروع كفالة الأيتام 2024',
                'idea' => 'مشروع شامل لكفالة ورعاية الأيتام يهدف إلى توفير الرعاية الكاملة للأطفال الأيتام من خلال برامج متكاملة تشمل التعليم والرعاية الصحية والنفسية والاجتماعية. المشروع يسعى لضمان حصول الأطفال على حقوقهم الأساسية في بيئة آمنة ومحفزة للنمو والتطور.',
                'grant_area_id' => $grantAreas->where('name', 'LIKE', '%أيتام%')->first()?->id ?? $grantAreas->first()->id,
                'lat' => 24.7136,
                'lng' => 46.6753,
                'start_date' => now()->addMonths(2),
                'duration_months' => 12,
                'target_group_type' => 'الأطفال الأيتام من عمر 5-15 سنة',
                'state_id' => $states->first()->id,
                'city_id' => $cities->first()->id,
                'beneficiaries_count' => 150,
                'goals' => 'توفير الرعاية الشاملة لـ 150 يتيم، تحسين مستوى التعليم بنسبة 80%، توفير الرعاية الصحية الدورية، تطوير المهارات الاجتماعية والنفسية للأطفال.',
                'strengths' => 'فريق عمل متخصص ومدرب، شراكات مع مؤسسات تعليمية وصحية، خبرة واسعة في مجال رعاية الأيتام، بنية تحتية قوية ومجهزة.',
                'risks' => 'التحديات المالية المحتملة، صعوبة في الحصول على التراخيص اللازمة، التحديات في توظيف الكوادر المتخصصة.',
                'declaration_checked' => true,
                'association_id' => $associations->first()->id,
                'created_by' => $users->first()?->id,
                'org_id' => $org->id,
            ],
            [
                'name' => 'برنامج الرعاية الصحية المجتمعية',
                'idea' => 'برنامج صحي شامل يهدف إلى توفير الخدمات الصحية الأساسية للمجتمعات المحتاجة. يشمل البرنامج الفحوصات الدورية، التطعيمات، العلاج الأساسي، والتوعية الصحية. كما يركز على الوقاية من الأمراض المزمنة وتعزيز الصحة العامة.',
                'grant_area_id' => $grantAreas->where('name', 'LIKE', '%صحية%')->first()?->id ?? $grantAreas->skip(1)->first()->id,
                'lat' => 21.4858,
                'lng' => 39.1925,
                'start_date' => now()->addMonth(),
                'duration_months' => 18,
                'target_group_type' => 'الأسر محدودة الدخل وكبار السن',
                'state_id' => $states->first()->id,
                'city_id' => $cities->first()->id,
                'beneficiaries_count' => 500,
                'goals' => 'خدمة 500 مستفيد شهرياً، تقليل معدل الأمراض المزمنة بنسبة 30%، رفع مستوى الوعي الصحي في المجتمع، توفير العلاج المجاني للحالات الطارئة.',
                'strengths' => 'شراكة مع المستشفيات المحلية، فريق طبي متخصص، معدات طبية حديثة، برامج توعية فعالة.',
                'risks' => 'نقص في المستلزمات الطبية، صعوبة في الوصول للمناطق النائية، تقلبات في أسعار الأدوية.',
                'declaration_checked' => true,
                'association_id' => $associations->first()->id,
                'created_by' => $users->first()?->id,
                'org_id' => $org->id,
            ],
            [
                'name' => 'مبادرة التعليم للجميع',
                'idea' => 'مبادرة تعليمية تهدف إلى محو الأمية وتوفير التعليم الأساسي للكبار والأطفال المتسربين من التعليم. تشمل المبادرة برامج تعليمية مخصصة، دورات مهنية، وبرامج لتطوير المهارات الحياتية.',
                'grant_area_id' => $grantAreas->first()->id,
                'lat' => 26.4207,
                'lng' => 50.0888,
                'start_date' => now()->addWeeks(3),
                'duration_months' => 24,
                'target_group_type' => 'الكبار الأميون والأطفال المتسربون',
                'state_id' => $states->first()->id,
                'city_id' => $cities->first()->id,
                'beneficiaries_count' => 300,
                'goals' => 'محو أمية 200 شخص بالغ، إعادة دمج 100 طفل في النظام التعليمي، توفير 50 دورة مهنية متخصصة.',
                'strengths' => 'منهج تعليمي مبتكر، مدرسون متخصصون، مرافق تعليمية مجهزة، دعم مجتمعي قوي.',
                'risks' => 'انخفاض معدلات الحضور، صعوبة في تحفيز المتعلمين الكبار، التحديات اللوجستية.',
                'declaration_checked' => true,
                'association_id' => $associations->first()->id,
                'created_by' => $users->first()?->id,
                'org_id' => $org->id,
            ],
        ];

        foreach ($orders as $orderData) {
            $order = Order::create($orderData);

            // Create deliverables for each order
            $this->createDeliverablesForOrder($order);
        }

        $this->command->info('تم إنشاء '.count($orders).' أمر مع المخرجات');
    }

    /**
     * Create deliverables for a given order
     */
    private function createDeliverablesForOrder(Order $order): void
    {
        $deliverables = [];
        $startDate = Carbon::parse($order->start_date);

        // Define deliverables based on order type/name
        if (str_contains($order->name, 'كفالة الأيتام')) {
            $deliverables = [
                [
                    'name' => 'برنامج الرعاية التعليمية',
                    'start_date' => $startDate,
                    'end_date' => $startDate->copy()->addMonths(6),
                    'amount' => 150000.00,
                    'notes' => 'توفير التعليم الأساسي والمواد التعليمية للأطفال الأيتام',
                ],
                [
                    'name' => 'برنامج الرعاية الصحية',
                    'start_date' => $startDate->copy()->addMonth(),
                    'end_date' => $startDate->copy()->addMonths(8),
                    'amount' => 85000.00,
                    'notes' => 'الفحوصات الدورية والعلاج الطبي والتطعيمات اللازمة',
                ],
                [
                    'name' => 'برنامج الأنشطة الترفيهية والاجتماعية',
                    'start_date' => $startDate->copy()->addMonths(2),
                    'end_date' => $startDate->copy()->addMonths(10),
                    'amount' => 45000.00,
                    'notes' => 'أنشطة ترفيهية وثقافية ورياضية لتطوير شخصية الأطفال',
                ],
                [
                    'name' => 'الدعم النفسي والاجتماعي',
                    'start_date' => $startDate,
                    'end_date' => $startDate->copy()->addMonths(12),
                    'amount' => 65000.00,
                    'notes' => 'جلسات الدعم النفسي والإرشاد الاجتماعي للأطفال',
                ],
            ];
        } elseif (str_contains($order->name, 'الرعاية الصحية')) {
            $deliverables = [
                [
                    'name' => 'عيادة متنقلة للمناطق النائية',
                    'start_date' => $startDate,
                    'end_date' => $startDate->copy()->addMonths(12),
                    'amount' => 200000.00,
                    'notes' => 'تشغيل عيادة متنقلة مجهزة بالكامل لخدمة المناطق النائية',
                ],
                [
                    'name' => 'برنامج الفحوصات الدورية',
                    'start_date' => $startDate->copy()->addWeeks(2),
                    'end_date' => $startDate->copy()->addMonths(15),
                    'amount' => 120000.00,
                    'notes' => 'فحوصات شاملة ودورية للكشف المبكر عن الأمراض',
                ],
                [
                    'name' => 'حملات التوعية الصحية',
                    'start_date' => $startDate->copy()->addMonth(),
                    'end_date' => $startDate->copy()->addMonths(16),
                    'amount' => 55000.00,
                    'notes' => 'حملات توعوية في المجتمع حول الصحة العامة والوقاية',
                ],
                [
                    'name' => 'توفير الأدوية والمستلزمات الطبية',
                    'start_date' => $startDate,
                    'end_date' => $startDate->copy()->addMonths(18),
                    'amount' => 180000.00,
                    'notes' => 'توفير الأدوية الأساسية والمستلزمات الطبية اللازمة',
                ],
            ];
        } elseif (str_contains($order->name, 'التعليم')) {
            $deliverables = [
                [
                    'name' => 'برنامج محو الأمية للكبار',
                    'start_date' => $startDate,
                    'end_date' => $startDate->copy()->addMonths(18),
                    'amount' => 95000.00,
                    'notes' => 'فصول دراسية لتعليم القراءة والكتابة للكبار الأميين',
                ],
                [
                    'name' => 'إعادة دمج الأطفال المتسربين',
                    'start_date' => $startDate->copy()->addMonth(),
                    'end_date' => $startDate->copy()->addMonths(20),
                    'amount' => 110000.00,
                    'notes' => 'برامج تعليمية مكثفة لإعادة دمج الأطفال في النظام التعليمي',
                ],
                [
                    'name' => 'الدورات المهنية والتقنية',
                    'start_date' => $startDate->copy()->addMonths(6),
                    'end_date' => $startDate->copy()->addMonths(22),
                    'amount' => 135000.00,
                    'notes' => 'دورات في الحرف اليدوية والمهن التقنية لتمكين المستفيدين',
                ],
                [
                    'name' => 'المواد التعليمية والمعدات',
                    'start_date' => $startDate,
                    'end_date' => $startDate->copy()->addMonths(24),
                    'amount' => 75000.00,
                    'notes' => 'توفير الكتب والمواد التعليمية والمعدات اللازمة للتدريب',
                ],
            ];
        } else {
            // Default deliverables for any other order type
            $deliverables = [
                [
                    'name' => 'المرحلة التحضيرية',
                    'start_date' => $startDate,
                    'end_date' => $startDate->copy()->addMonths(2),
                    'amount' => 50000.00,
                    'notes' => 'إعداد وتجهيز المتطلبات الأساسية للمشروع',
                ],
                [
                    'name' => 'مرحلة التنفيذ الأساسية',
                    'start_date' => $startDate->copy()->addMonths(2),
                    'end_date' => $startDate->copy()->addMonths(8),
                    'amount' => 150000.00,
                    'notes' => 'تنفيذ الأنشطة الرئيسية للمشروع وتقديم الخدمات المطلوبة',
                ],
                [
                    'name' => 'مرحلة المتابعة والتقييم',
                    'start_date' => $startDate->copy()->addMonths(8),
                    'end_date' => $startDate->copy()->addMonths(12),
                    'amount' => 35000.00,
                    'notes' => 'متابعة النتائج وتقييم الأثر ووضع التوصيات',
                ],
            ];
        }

        // Create the deliverables
        foreach ($deliverables as $deliverableData) {
            OrderDeliverable::create([
                'order_id' => $order->id,
                'name' => $deliverableData['name'],
                'start_date' => $deliverableData['start_date'],
                'end_date' => $deliverableData['end_date'],
                'amount' => $deliverableData['amount'],
                'notes' => $deliverableData['notes'],
            ]);
        }
    }
}
