<?php

namespace Database\Seeders;

use App\Models\Initiative;
use App\Models\InitiativeOperationGoal;
use App\Models\User;
use Illuminate\Database\Seeder;

class InitiativeOperationGoalSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $user = User::first();
        $initiatives = Initiative::all();

        if ($initiatives->isEmpty()) {
            $this->command->warn('No initiatives found. Please run InitiativeSeeder first.');

            return;
        }

        // Define operation goals templates based on initiative types
        $operationGoalTemplates = [
            // For training/development initiatives
            'تدريب' => [
                ['name' => 'عدد المتدربين المستهدفين', 'criteria' => 'عدد الأشخاص المسجلين في البرنامج التدريبي'],
                ['name' => 'معدل إكمال التدريب', 'criteria' => 'نسبة المتدربين الذين أكملوا البرنامج بنجاح'],
                ['name' => 'مستوى رضا المتدربين', 'criteria' => 'متوسط تقييم المتدربين للبرنامج من 10'],
            ],
            'أكاديمية' => [
                ['name' => 'عدد الخريجين المؤهلين', 'criteria' => 'عدد المشاركين الذين اجتازوا جميع متطلبات الأكاديمية'],
                ['name' => 'معدل التوظيف بعد التخرج', 'criteria' => 'نسبة الخريجين الذين حصلوا على فرص عمل'],
                ['name' => 'تقييم جودة التدريب', 'criteria' => 'تقييم المدربين والخبراء للمحتوى التدريبي'],
            ],
            // For support initiatives
            'مساعدات' => [
                ['name' => 'عدد الأسر المستفيدة', 'criteria' => 'إجمالي عدد الأسر التي تلقت المساعدات'],
                ['name' => 'قيمة المساعدات الموزعة', 'criteria' => 'إجمالي المبلغ المالي للمساعدات المقدمة'],
                ['name' => 'سرعة الاستجابة للطوارئ', 'criteria' => 'متوسط الوقت بالأيام للاستجابة للحالات الطارئة'],
            ],
            'سلال' => [
                ['name' => 'عدد السلال الموزعة شهرياً', 'criteria' => 'إجمالي عدد السلال الغذائية الموزعة'],
                ['name' => 'تنوع المواد الغذائية', 'criteria' => 'عدد أنواع المواد الغذائية المختلفة في كل سلة'],
                ['name' => 'جودة المواد الغذائية', 'criteria' => 'نسبة المواد الغذائية الطازجة وعالية الجودة'],
            ],
            // For empowerment initiatives
            'حاضنة' => [
                ['name' => 'عدد المشاريع المحتضنة', 'criteria' => 'إجمالي عدد المشاريع الشبابية المدعومة'],
                ['name' => 'معدل نجاح المشاريع', 'criteria' => 'نسبة المشاريع التي استمرت أكثر من سنة'],
                ['name' => 'قيمة الاستثمار المولد', 'criteria' => 'إجمالي رؤوس الأموال المجمعة للمشاريع'],
            ],
            // For investment initiatives
            'استثمار' => [
                ['name' => 'العائد على الاستثمار', 'criteria' => 'نسبة الأرباح المحققة مقارنة برأس المال المستثمر'],
                ['name' => 'عدد المشاريع الاستثمارية', 'criteria' => 'إجمالي عدد المشاريع في المحفظة الاستثمارية'],
                ['name' => 'مستوى المخاطر', 'criteria' => 'تقييم درجة المخاطر للمحفظة الاستثمارية من 10'],
            ],
            'طاقة' => [
                ['name' => 'قدرة الطاقة المنتجة', 'criteria' => 'كمية الطاقة المتجددة المنتجة بالكيلوواط/ساعة'],
                ['name' => 'توفير التكاليف', 'criteria' => 'نسبة التوفير في فواتير الكهرباء'],
                ['name' => 'تقليل الانبعاثات الكربونية', 'criteria' => 'كمية ثاني أكسيد الكربون المتجنبة بالطن'],
            ],
            // For innovation initiatives
            'مختبر' => [
                ['name' => 'عدد الحلول المبتكرة', 'criteria' => 'إجمالي عدد الأفكار والحلول المطورة'],
                ['name' => 'معدل تطبيق الحلول', 'criteria' => 'نسبة الحلول التي تم تطبيقها عملياً'],
                ['name' => 'مستوى الابتكار', 'criteria' => 'تقييم درجة الإبداع والحداثة للحلول المطورة'],
            ],
        ];

        $createdGoals = [];

        foreach ($initiatives as $initiative) {
            // Determine which template to use based on initiative name
            $selectedTemplate = null;
            foreach ($operationGoalTemplates as $keyword => $template) {
                if (str_contains($initiative->name, $keyword)) {
                    $selectedTemplate = $template;
                    break;
                }
            }

            // If no specific template found, use a generic one
            if (! $selectedTemplate) {
                $selectedTemplate = [
                    ['name' => 'معدل تحقيق الأهداف', 'criteria' => 'نسبة الأهداف المحققة من إجمالي الأهداف المخططة'],
                    ['name' => 'مستوى جودة التنفيذ', 'criteria' => 'تقييم جودة تنفيذ أنشطة المبادرة من 10'],
                ];
            }

            // Create 2-3 operation goals per initiative
            $goalCount = min(rand(2, 3), count($selectedTemplate));
            $selectedGoals = fake()->randomElements($selectedTemplate, $goalCount);

            foreach ($selectedGoals as $goalTemplate) {
                $target = fake()->numberBetween(10, 100);
                $actual = $this->generateActualValue($target, 0.15); // 15% variance

                $goal = InitiativeOperationGoal::create([
                    'initiative_id' => $initiative->id,
                    'name' => $goalTemplate['name'],
                    'performance_criteria' => $goalTemplate['criteria'],
                    'target' => $target,
                    'actual' => $actual,
                    'created_by' => $user->id,
                ]);

                $createdGoals[] = $goal;
            }
        }

        $this->command->info('تم إنشاء '.count($createdGoals).' هدف عملياتي لـ '.$initiatives->count().' مبادرة');
    }

    /**
     * Generate actual value with some variance from target value
     */
    private function generateActualValue(float $target, float $variance): float
    {
        $minVariance = 1 - $variance;
        $maxVariance = 1 + $variance;
        $multiplier = fake()->randomFloat(2, $minVariance, $maxVariance);

        return round($target * $multiplier, 2);
    }
}
