# WorkflowTabs Usage Guide

The `WorkflowTabs` class provides a generalized way to add workflow-based tabs to any Filament List page for models that use the `HasWorkflow` trait.

## Basic Usage

### Simple Implementation

```php
<?php

namespace App\Filament\Admin\Resources\Orders\Pages;

use App\Filament\Admin\Resources\Orders\OrderResource;
use App\Models\Order;
use App\Workflows\Actions\WorkflowTabs;
use Filament\Actions\CreateAction;
use Filament\Resources\Pages\ListRecords;

class ListOrders extends ListRecords
{
    protected static string $resource = OrderResource::class;

    protected function getHeaderActions(): array
    {
        return [
            CreateAction::make(),
        ];
    }

    public function getTabs(): array
    {
        return WorkflowTabs::getTabs(Order::class);
    }
}
```

This will automatically generate tabs for:
- **All**: All records
- **Workflow Steps**: Dynamic tabs for each step in the active workflow template
- **Completed**: Records with completed workflows
- **Rejected**: Records with rejected workflows
- **No Workflow**: Records without workflow instances

## Advanced Usage

### Custom Tab Selection

```php
public function getTabs(): array
{
    // Exclude specific tabs
    return WorkflowTabs::getTabsWithOptions(Order::class, [
        'exclude' => ['rejected', 'no_workflow']
    ]);
}
```

```php
public function getTabs(): array
{
    // Include only specific tabs
    return WorkflowTabs::getTabsWithOptions(Order::class, [
        'include' => ['all', 'study', 'approval', 'completed']
    ]);
}
```

### Step-Only Tabs

```php
public function getTabs(): array
{
    // Only show workflow step tabs (no status tabs)
    return WorkflowTabs::getStepTabs(Order::class);
}
```

### Status-Only Tabs

```php
public function getTabs(): array
{
    // Only show status tabs (All, Completed, Rejected, No Workflow)
    return WorkflowTabs::getStatusTabs(Order::class);
}
```

### Combined Approach

```php
public function getTabs(): array
{
    $tabs = [];
    
    // Add custom "Recent" tab
    $tabs['recent'] = Tab::make('Recent')
        ->badge(Order::where('created_at', '>=', now()->subDays(7))->count())
        ->modifyQueryUsing(fn (Builder $query) => 
            $query->where('created_at', '>=', now()->subDays(7))
        );
    
    // Add workflow tabs
    $tabs = array_merge($tabs, WorkflowTabs::getTabs(Order::class));
    
    return $tabs;
}
```

## Working with Different Models

The `WorkflowTabs` class works with any model that implements the `HasWorkflow` trait:

```php
// For Orders
WorkflowTabs::getTabs(Order::class);

// For ProjectCharters  
WorkflowTabs::getTabs(ProjectCharter::class);

// For any other model with workflows
WorkflowTabs::getTabs(YourModel::class);
```

## Tab Features

Each generated tab includes:

- **Badge Counts**: Real-time count of records in each category
- **Query Filtering**: Automatic filtering of the table based on workflow state
- **Dynamic Steps**: Tabs are generated from active published workflow templates
- **Internationalization**: Uses Laravel translation keys for labels
- **Efficient Queries**: Optimized database queries for performance

## Customization

### Tab Labels

Tab labels use Laravel's translation system:
- Workflow states: `__('workflow.states.completed')`
- Workflow status: `__('workflow.status.no_workflow')`
- Step labels: Taken from workflow template configuration

### Badge Colors

To customize badge colors, you can extend the tabs after generation:

```php
public function getTabs(): array
{
    $tabs = WorkflowTabs::getTabs(Order::class);
    
    // Customize specific tabs
    $tabs['rejected']->badgeColor('danger');
    $tabs['completed']->badgeColor('success');
    
    return $tabs;
}
```

## Requirements

1. Model must use the `HasWorkflow` trait
2. Active published workflow template must exist for the model
3. Proper Laravel translation keys should be configured
4. Model must be properly configured in `config/workflows.php`

## Benefits

- **DRY Principle**: Single implementation for all workflow models
- **Consistency**: Same tab behavior across all resources
- **Maintainability**: Changes in one place affect all implementations
- **Flexibility**: Multiple usage patterns available
- **Performance**: Efficient database queries
- **Internationalization**: Built-in translation support