<?php

namespace App\Models;

use App\Enums\ProjectCharterPhase;
use App\Enums\ProjectCharterState;
use App\Traits\BelongsToOrg;
use App\Workflows\Traits\HasWorkflow;
use Carbon\Carbon;
use Database\Factories\ProjectCharterFactory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Support\Facades\Auth;
use Spatie\Activitylog\LogOptions;
use Spatie\Activitylog\Traits\LogsActivity;

class ProjectCharter extends Model
{
    /** @use HasFactory<ProjectCharterFactory> */
    use BelongsToOrg, HasFactory, HasWorkflow, LogsActivity;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'initiative_project_id',
        'initiative_id',
        'name',
        'description',
        'state',
        'start_date',
        'expected_end_date',
        'total_cost',
        'created_by',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'start_date' => 'date',
            'expected_end_date' => 'date',
            'total_cost' => 'decimal:2',
            'state' => ProjectCharterState::class,
        ];
    }

    /**
     * Get the initiative project that this charter belongs to.
     */
    public function initiativeProject(): BelongsTo
    {
        return $this->belongsTo(InitiativeProject::class);
    }

    /**
     * Get the initiative that this charter belongs to.
     */
    public function initiative(): BelongsTo
    {
        return $this->belongsTo(Initiative::class);
    }

    /**
     * Get the user who created this charter.
     */
    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the deliverables for this charter.
     */
    public function deliverables(): HasMany
    {
        return $this->hasMany(ProjectCharterDeliverable::class);
    }

    /**
     * Get the tasks through deliverables.
     */
    public function tasks(): HasManyThrough
    {
        return $this->hasManyThrough(ProjectCharterDeliverableTask::class, ProjectCharterDeliverable::class);
    }

    /**
     * Get the risks for this charter.
     */
    public function risks(): HasMany
    {
        return $this->hasMany(ProjectCharterRisk::class);
    }

    /**
     * Get the files for this charter.
     */
    public function files(): HasMany
    {
        return $this->hasMany(ProjectCharterFile::class);
    }

    /**
     * Should show project charters progress views only when project state is not active, or workflow is completed
     */
    public function shouldShowProgressView(): bool
    {
        return $this->state !== ProjectCharterState::ACTIVE
            || ($this->hasWorkflow() && $this->workflowInstance->isCompleted());
    }

    /**
     * get the ProjectCharterPhase based on its status
     */
    public function getPhaseAttribute(): ProjectCharterPhase
    {
        return match (true) {
            in_array($this->state, [
                ProjectCharterState::COMPLETED,
                ProjectCharterState::CANCELED,
                ProjectCharterState::SUSPENDED,
            ]) || ($this->hasWorkflow() && $this->workflowInstance->isRejected()) => ProjectCharterPhase::CLOSURE,

            $this->state === ProjectCharterState::ACTIVE
                && ($this->hasWorkflow() && ! $this->workflowInstance->isCompleted()) => ProjectCharterPhase::INITIATION,

            default => ProjectCharterPhase::EXECUTION,
        };
    }

    /**
     * Get the estimated progress percentage based on timeline
     */
    public function getEstimatedProgressPercentageAttribute(): float
    {
        if (! $this->start_date || ! $this->expected_end_date) {
            return 0;
        }

        $startDate = Carbon::parse($this->start_date);
        $endDate = Carbon::parse($this->expected_end_date);
        $currentDate = Carbon::now();

        if ($currentDate <= $startDate) {
            return 0;
        }

        if ($currentDate >= $endDate) {
            return 100;
        }

        $totalDays = $startDate->diffInDays($endDate);
        $elapsedDays = $startDate->diffInDays($currentDate);

        return ($elapsedDays / $totalDays) * 100;
    }

    public function getProjectProgressAttribute(): float
    {
        $deliverables = $this->deliverables;

        if ($deliverables->isEmpty()) {
            return 0;
        }

        $totalProgress = $deliverables->sum('progress_percentage');

        return $totalProgress / $deliverables->count();
    }

    /**
     * Get the progress deviation (actual - estimated)
     */
    public function getProgressDeviationAttribute(): float
    {
        return $this->project_progress - $this->estimated_progress_percentage;
    }

    /**
     * Get the financial progress percentage based on approved exchange requests vs estimated cost
     */
    public function getFinancialProgressAttribute(): float
    {
        if (! $this->total_cost || $this->total_cost <= 0) {
            return 0;
        }

        $approvedValue = $this->total_approved_exchange_requests;

        return min(100, ($approvedValue / $this->total_cost) * 100);
    }

    /**
     * Get the total value of all approved exchange requests for this charter
     */
    public function getTotalApprovedExchangeRequestsAttribute(): float
    {
        return $this->deliverables()
            ->with(['completionReports.exchangeRequest'])
            ->get()
            ->pluck('completionReports')
            ->flatten()
            ->pluck('exchangeRequest')
            // TODO: filter must use Workflow
            // ->filter(function ($exchangeRequest) {
            //     return $exchangeRequest && $exchangeRequest->status === \App\Models\ExchangeRequest::STATUS_APPROVED;
            // })
            ->sum(function ($exchangeRequest) {
                return $exchangeRequest->completionReport->value ?? 0;
            });
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (Auth::check() && ! $model->created_by) {
                $model->created_by = Auth::id();
            }
        });
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->useLogName('project_charter');
    }
}
