<?php

namespace App\Filament\Association\Resources\Orders\Schemas;

use App\Models\Association;
use App\Models\City;
use App\Models\GrantArea;
use App\Models\State;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\MarkdownEditor;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Wizard;
use Filament\Schemas\Components\Wizard\Step;
use Filament\Schemas\Schema;

class OrderForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                Wizard::make([
                    Step::make(__('order.project_basics'))
                        ->description(__('order.project_basics_description'))
                        ->schema([
                            Section::make(__('order.project_information'))
                                ->schema([
                                    TextInput::make('name')
                                        ->label(__('order.project_name'))
                                        ->placeholder(__('order.project_name_placeholder'))
                                        ->maxLength(200)
                                        ->required()
                                        ->columnSpanFull(),

                                    MarkdownEditor::make('idea')
                                        ->label(__('order.project_idea'))
                                        ->placeholder(__('order.project_idea_placeholder'))
                                        ->maxLength(5000)
                                        ->required()
                                        ->columnSpanFull(),

                                    Select::make('grant_area_id')
                                        ->label(__('order.grant_area'))
                                        ->placeholder(__('order.select_grant_area'))
                                        ->options(GrantArea::pluck('name', 'id'))
                                        ->searchable()
                                        ->required(),

                                    Select::make('association_id')
                                        ->label(__('order.association'))
                                        ->placeholder(__('order.select_association'))
                                        ->options(Association::pluck('name', 'id'))
                                        ->searchable()
                                        ->required(),
                                ])
                                ->columns(2),
                        ]),

                    Step::make(__('order.project_details'))
                        ->description(__('order.project_details_description'))
                        ->schema([
                            Section::make(__('order.timeline_and_scope'))
                                ->schema([
                                    Grid::make(2)
                                        ->schema([
                                            DatePicker::make('start_date')
                                                ->label(__('order.start_date'))
                                                ->placeholder(__('order.select_start_date'))
                                                ->required()
                                                ->minDate(now()->addDay()),

                                            TextInput::make('duration_months')
                                                ->label(__('order.duration_months'))
                                                ->placeholder(__('order.duration_placeholder'))
                                                ->numeric()
                                                ->minValue(1)
                                                ->maxValue(60)
                                                ->required(),
                                        ]),

                                    TextInput::make('target_group_type')
                                        ->label(__('order.target_group_type'))
                                        ->placeholder(__('order.target_group_placeholder'))
                                        ->maxLength(120)
                                        ->required(),

                                    TextInput::make('beneficiaries_count')
                                        ->label(__('order.beneficiaries_count'))
                                        ->placeholder(__('order.beneficiaries_placeholder'))
                                        ->numeric()
                                        ->minValue(1)
                                        ->required(),
                                ])
                                ->columns(1),

                            Section::make(__('order.location'))
                                ->schema([
                                    Select::make('state_id')
                                        ->label(__('order.state'))
                                        ->placeholder(__('order.select_state'))
                                        ->options(State::pluck('name', 'id'))
                                        ->searchable()
                                        ->required()
                                        ->live()
                                        ->afterStateUpdated(fn (callable $set) => $set('city_id', null)),

                                    Select::make('city_id')
                                        ->label(__('order.city'))
                                        ->placeholder(__('order.select_city'))
                                        ->options(fn (Get $get): array => City::where('state_id', $get('state_id'))->pluck('name', 'id')->toArray()
                                        )
                                        ->searchable()
                                        ->required()
                                        ->disabled(fn (Get $get): bool => ! $get('state_id')),

                                    Grid::make(2)
                                        ->schema([
                                            TextInput::make('lat')
                                                ->label(__('order.latitude'))
                                                ->placeholder(__('order.latitude_placeholder'))
                                                ->numeric()
                                                ->step(0.00000001)
                                                ->minValue(-90)
                                                ->maxValue(90)
                                                ->required(),

                                            TextInput::make('lng')
                                                ->label(__('order.longitude'))
                                                ->placeholder(__('order.longitude_placeholder'))
                                                ->numeric()
                                                ->step(0.00000001)
                                                ->minValue(-180)
                                                ->maxValue(180)
                                                ->required(),
                                        ]),
                                ])
                                ->columns(2),
                        ]),

                    Step::make(__('order.project_content'))
                        ->description(__('order.project_content_description'))
                        ->schema([
                            Section::make(__('order.project_analysis'))
                                ->schema([
                                    MarkdownEditor::make('goals')
                                        ->label(__('order.goals'))
                                        ->placeholder(__('order.goals_placeholder'))
                                        ->maxLength(5000)
                                        ->required()
                                        ->columnSpanFull(),

                                    MarkdownEditor::make('strengths')
                                        ->label(__('order.strengths'))
                                        ->placeholder(__('order.strengths_placeholder'))
                                        ->maxLength(5000)
                                        ->required()
                                        ->columnSpanFull(),

                                    MarkdownEditor::make('risks')
                                        ->label(__('order.risks'))
                                        ->placeholder(__('order.risks_placeholder'))
                                        ->maxLength(5000)
                                        ->required()
                                        ->columnSpanFull(),
                                ])
                                ->columns(1),
                        ]),

                    Step::make(__('order.deliverables'))
                        ->description(__('order.deliverables_description'))
                        ->schema([
                            Section::make(__('order.project_deliverables'))
                                ->schema([
                                    Repeater::make('deliverables')
                                        ->label(__('order.deliverables'))
                                        ->relationship()
                                        ->schema([
                                            TextInput::make('name')
                                                ->label(__('order.deliverable_name'))
                                                ->placeholder(__('order.deliverable_name_placeholder'))
                                                ->maxLength(200)
                                                ->required()
                                                ->columnSpan(2),

                                            DatePicker::make('start_date')
                                                ->label(__('order.start_date'))
                                                ->required(),

                                            DatePicker::make('end_date')
                                                ->label(__('order.end_date'))
                                                ->required()
                                                ->after('start_date'),

                                            TextInput::make('amount')
                                                ->label(__('order.amount'))
                                                ->placeholder(__('order.amount_placeholder'))
                                                ->numeric()
                                                ->minValue(0)
                                                ->step(0.01)
                                                ->prefix('SAR')
                                                ->required(),

                                            MarkdownEditor::make('notes')
                                                ->label(__('order.notes'))
                                                ->placeholder(__('order.notes_placeholder'))
                                                ->maxLength(2000)
                                                ->columnSpanFull(),
                                        ])
                                        ->columns(3)
                                        ->reorderable()
                                        ->collapsible()
                                        ->itemLabel(fn (array $state): ?string => $state['name'] ?? null)
                                        ->addActionLabel(__('order.add_deliverable'))
                                        ->minItems(1)
                                        ->maxItems(20),
                                ])
                                ->columns(1),
                        ]),

                    Step::make(__('order.attachments'))
                        ->description(__('order.attachments_description'))
                        ->schema([
                            Section::make(__('order.support_documents'))
                                ->schema([
                                    SpatieMediaLibraryFileUpload::make('support_letter')
                                        ->label(__('order.support_letter'))
                                        ->helperText(__('order.support_letter_help'))
                                        ->collection('support_letter')
                                        ->acceptedFileTypes(['application/pdf', 'image/jpeg', 'image/png'])
                                        ->maxSize(5120) // 5MB
                                        ->required(),

                                    SpatieMediaLibraryFileUpload::make('project_attachments')
                                        ->label(__('order.project_attachments'))
                                        ->helperText(__('order.project_attachments_help'))
                                        ->collection('project_attachments')
                                        ->multiple()
                                        ->acceptedFileTypes([
                                            'application/pdf',
                                            'application/vnd.ms-excel',
                                            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                                            'image/jpeg',
                                            'image/png',
                                            'image/gif',
                                            'text/plain',
                                            'application/msword',
                                            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                                        ])
                                        ->maxSize(5120) // 5MB per file
                                        ->maxFiles(10),
                                ])
                                ->columns(1),

                            Section::make(__('order.declaration'))
                                ->schema([
                                    Checkbox::make('declaration_checked')
                                        ->label(__('order.declaration_text'))
                                        ->helperText(__('order.declaration_help'))
                                        ->required()
                                        ->accepted(),
                                ])
                                ->columns(1),
                        ]),
                ])
                    ->columnSpanFull()
                    ->skippable(),
            ]);
    }
}
