<?php

namespace App\Filament\Admin\Resources\ProjectCharters\Pages;

use App\Filament\Admin\Resources\ProjectCharters\ProjectCharterResource;
use BackedEnum;
use Filament\Actions\AssociateAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\CreateAction;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\DissociateAction;
use Filament\Actions\DissociateBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ForceDeleteAction;
use Filament\Actions\ForceDeleteBulkAction;
use Filament\Actions\RestoreAction;
use Filament\Actions\RestoreBulkAction;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\TextInput;
use Filament\Resources\Pages\ManageRelatedRecords;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Support\Icons\Heroicon;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class ManageProjectCharterFiles extends ManageRelatedRecords
{
    protected static string $resource = ProjectCharterResource::class;

    protected static string $relationship = 'files';

    protected static string|BackedEnum|null $navigationIcon = Heroicon::OutlinedRectangleStack;

    public function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make(__('project_charter_file.file_details'))
                    ->description(__('project_charter_file.file_details_description'))
                    ->schema([
                        TextInput::make('name')
                            ->label(__('project_charter_file.name'))
                            ->required()
                            ->maxLength(255)
                            ->placeholder(__('project_charter_file.name_placeholder'))
                            ->columnSpanFull(),

                        SpatieMediaLibraryFileUpload::make('charter_file')
                            ->label(__('project_charter_file.file_upload'))
                            ->collection('charter_files')
                            ->acceptedFileTypes([
                                'application/pdf',
                                'image/jpeg',
                                'image/png',
                                'image/jpg',
                            ])
                            ->maxSize(10240) // 10MB
                            ->downloadable()
                            ->required()
                            ->helperText(__('project_charter_file.file_helper'))
                            ->columnSpanFull(),
                    ])
                    ->columns(1),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('name')
            ->columns([
                TextColumn::make('name')
                    ->label(__('project_charter_file.name'))
                    ->searchable()
                    ->sortable()
                    ->weight('medium'),

                TextColumn::make('file_name')
                    ->label(__('project_charter_file.file_name'))
                    ->getStateUsing(fn ($record) => $record->getFirstMedia('charter_files')?->name ?? '-')
                    ->icon('heroicon-o-document')
                    ->placeholder('-'),

                TextColumn::make('file_size')
                    ->label(__('project_charter_file.file_size'))
                    ->getStateUsing(fn ($record) => $record->file_size ?? '-')
                    ->placeholder('-'),

                // IconColumn::make('has_file')
                //     ->label(__('project_charter_file.has_file'))
                //     ->boolean()
                //     ->getStateUsing(fn ($record) => $record->hasFile()),

                TextColumn::make('creator.name')
                    ->label(__('project_charter_file.uploaded_by'))
                    ->placeholder('-')
                    ->toggleable(),

                TextColumn::make('created_at')
                    ->label(__('project_charter_file.uploaded_at'))
                    ->date()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                TrashedFilter::make(),
            ])
            ->headerActions([
                CreateAction::make(),
                AssociateAction::make(),
            ])
            ->recordActions([
                EditAction::make(),
                DissociateAction::make(),
                DeleteAction::make(),
                ForceDeleteAction::make(),
                RestoreAction::make(),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DissociateBulkAction::make(),
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->modifyQueryUsing(fn (Builder $query) => $query
                ->withoutGlobalScopes([
                    SoftDeletingScope::class,
                ]));
    }
}
