<?php

namespace App\Filament\Admin\Resources\Orders\RelationManagers;

use App\Enums\RequirementStatus;
use Filament\Actions\AssociateAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\CreateAction;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\DissociateAction;
use Filament\Actions\DissociateBulkAction;
use Filament\Actions\EditAction;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\TextInput;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Schemas\Schema;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Model;

class RequirementsRelationManager extends RelationManager
{
    protected static string $relationship = 'requirements';

    public static function getTitle(Model $ownerRecord, string $pageClass): string
    {
        return __('order_requirement.requirements');
    }

    public static function getModelLabel(): string
    {
        return __('order_requirement.requirement');
    }

    public static function getPluralModelLabel(): string
    {
        return __('order_requirement.requirements');
    }

    public function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                TextInput::make('name')
                    ->label(__('order_requirement.name'))
                    ->required()
                    ->maxLength(255),

                Hidden::make('status')
                    ->default(RequirementStatus::PENDING),

                SpatieMediaLibraryFileUpload::make('requirement_file')
                    ->label(__('order_requirement.requirement_file'))
                    ->collection('requirement_file')
                    ->directory('order-requirements')
                    ->acceptedFileTypes([
                        'application/pdf',
                        'image/jpeg',
                        'image/png',
                        'image/jpg',
                    ])
                    ->downloadable()
                    ->maxSize(10240) // 10MB
                    ->helperText(__('order_requirement.file_upload_helper')),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('name')
            ->columns([
                TextColumn::make('name')
                    ->label(__('order_requirement.name'))
                    ->sortable()
                    ->searchable(),

                TextColumn::make('status')
                    ->label(__('order_requirement.status'))
                    ->badge(),

                TextColumn::make('created_at')
                    ->label(__('order_requirement.created_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('status')
                    ->label(__('order_requirement.status'))
                    ->options(RequirementStatus::class),
            ])
            ->headerActions([
                CreateAction::make(),
                AssociateAction::make(),
            ])
            ->recordActions([
                EditAction::make(),
                DissociateAction::make(),
                DeleteAction::make(),
                // TODO: accept/reject/ask modification actions
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DissociateBulkAction::make(),
                    DeleteBulkAction::make(),
                ]),
            ]);
    }

    public function isReadOnly(): bool
    {
        return false;
    }
}
