<?php

namespace App\Filament\Admin\Resources\Associations\Tables;

use Filament\Actions\Action;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ForceDeleteBulkAction;
use Filament\Actions\RestoreBulkAction;
use Filament\Actions\ViewAction;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Auth;

class AssociationTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('id')
                    ->label(__('common.id'))
                    ->sortable()
                    ->toggleable(),

                TextColumn::make('name')
                    ->label(__('association.name'))
                    ->sortable()
                    ->searchable(),

                TextColumn::make('is_verified')
                    ->label(__('association.verification_status'))
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        '1' => 'success',
                        '0' => 'danger',
                        default => 'gray',
                    })
                    ->formatStateUsing(fn (string $state): string => $state ? __('association.verified') : __('association.not_verified'))
                    ->sortable()
                    ->toggleable(),
            ])
            ->filters([
                TrashedFilter::make(),
            ])
            ->recordActions([
                ViewAction::make(),
                EditAction::make(),

                Action::make('verify')
                    ->label(__('association.action_verify'))
                    ->icon('heroicon-o-shield-check')
                    ->visible(fn ($record) => ! $record->is_verified && Auth::user()->can('verify_association'))
                    ->color('success')
                    ->requiresConfirmation()
                    ->modalHeading(__('association.confirm_verify_title'))
                    ->modalDescription(__('association.confirm_verify_description'))
                    ->action(fn ($record) => $record->update(['is_verified' => true])),

                Action::make('unverify')
                    ->label(__('association.action_unverify'))
                    ->icon('heroicon-o-shield-exclamation')
                    ->visible(fn ($record) => $record->is_verified && Auth::user()->can('unverify_association'))
                    ->color('warning')
                    ->requiresConfirmation()
                    ->modalHeading(__('association.confirm_unverify_title'))
                    ->modalDescription(__('association.confirm_unverify_description'))
                    ->action(fn ($record) => $record->update(['is_verified' => false])),

                Action::make('lock')
                    ->label(__('association.action_lock'))
                    ->icon('heroicon-o-lock-closed')
                    ->visible(fn ($record) => ! $record->is_locked && Auth::user()->can('lock_profile_association'))
                    ->color('danger')
                    ->requiresConfirmation()
                    ->modalHeading(__('association.confirm_lock_title'))
                    ->modalDescription(__('association.confirm_lock_description'))
                    ->action(fn ($record) => $record->update(['is_locked' => true])),

                Action::make('unlock')
                    ->label(__('association.action_unlock'))
                    ->icon('heroicon-o-lock-open')
                    ->visible(fn ($record) => $record->is_locked && Auth::user()->can('unlock_profile_association'))
                    ->color('success')
                    ->requiresConfirmation()
                    ->modalHeading(__('association.confirm_unlock_title'))
                    ->modalDescription(__('association.confirm_unlock_description'))
                    ->action(fn ($record) => $record->update(['is_locked' => false])),

                DeleteAction::make(),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ]);
    }
}
